/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.api.core;

import static java.util.function.Predicate.not;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource.Diagnostic;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil.Bound;
import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression;
import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression.Language;
import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.sysmlv2.sysMLv2.ActionUsage;
import eu.fbk.sysmlv2.sysMLv2.Assignment;
import eu.fbk.sysmlv2.sysMLv2.AttributeUsage;
import eu.fbk.sysmlv2.sysMLv2.Calculation;
import eu.fbk.sysmlv2.sysMLv2.CalculationUsage;
import eu.fbk.sysmlv2.sysMLv2.ConnectionElement;
import eu.fbk.sysmlv2.sysMLv2.ConstraintUsage;
import eu.fbk.sysmlv2.sysMLv2.Container;
import eu.fbk.sysmlv2.sysMLv2.Definition;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Direction;
import eu.fbk.sysmlv2.sysMLv2.EnumDefinition;
import eu.fbk.sysmlv2.sysMLv2.Expression;
import eu.fbk.sysmlv2.sysMLv2.Feature;
import eu.fbk.sysmlv2.sysMLv2.Model;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.Part;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.SendUsage;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.Transition;
import eu.fbk.sysmlv2.sysMLv2.TypedElement;
import eu.fbk.sysmlv2.sysMLv2.Usage;
import eu.fbk.sysmlv2.sysMLv2.Value;
import eu.fbk.sysmlv2.sysMLv2.impl.PartUsageImpl;
import eu.fbk.sysmlv2.util.SysMLv2FeatureChainSwitch;
import eu.fbk.sysmlv2.util.SysMLv2Util;

/**
 * This class contains all the available methods to navigate the SysMLv2 model.
 * The model should be already loaded and parsed. Nodes of the model AST will
 * likely be the arguments of the methods.
 */
public class ModelLibrary {
	private static final SysMLv2SystemModel SYSTEM_MODEL = new SysMLv2SystemModel();
	private static final SysMLv2StateMachineModel STATE_MACHINE_MODEL = new SysMLv2StateMachineModel();
	private static final String PREEFFECT = "preeffect";
	private static final String POSTEFFECT = "posteffect";
	private static final String PRECONDITION = "precondition";
	private static final String POSTCONDITION = "postcondition";
	private static final String REAL_ASSET = "realAsset";
	private static final String ENVIRONMENT = "env";
	private static final String ACTIVITY_PARAM = "ActivityParam";
	private static final String INVARIANT = "invariant";
	private static final String GET_LOWER_BOUND_DURATION = "getLowerBoundDuration";
	private static final String GET_UPPER_BOUND_DURATION = "getUpperBoundDuration";
	private static final String UNPLANNABLE = "Unplannable";

	/**
	 * Returns the list of parsing, linking and validation errors present in the model. If the list is empty, the model is valid.
	 * @param systemComponent the model's root component
	 * @return the list of errors
	 */
	public List<Diagnostic> validateModel(PartUsage systemComponent) {
		final ResourceSet resourceSet = EcoreUtil2.getResourceSet(systemComponent);
		final List<Diagnostic> errors = new ArrayList<>();
		resourceSet.getResources().forEach(resource -> errors.addAll(resource.getErrors()));
		return errors;
	}

	/**
	 * Returns a list of names of the subcomponents of the given part usage.
	 * @param component the owner part usage
	 * @return the names of the subcomponents
	 */
	public List<String> getSubComponentsNames(PartUsage component) {
		return getSubComponents(component).stream().map(PartUsage::getName).toList();
	}

	/**
	 * Returns the list of transitions that are not initial in the given state machine.
	 * @param stateMachine the state machine
	 * @return the list of transitions
	 */
	public List<Transition> getNonInitTransitions(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getNonInitTransitions(stateMachine);
	}

	/**
	 * Returns the names of transitions that are not initial in the given state machine.
	 * @param stateMachine the state machine
	 * @return the names of the transitions
	 */
	public List<String> getNonInitTransitionsNameList(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getNonInitTransitionsNameList(stateMachine);
	}

	/**
	 * Returns the list of initial transitions in the given state machine.
	 * @param stateMachine the state machine
	 * @return the list of transitions
	 */
	public List<Transition> getInitTransitions(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getInitTransitions(stateMachine);
	}

	/**
	 * Returns the list of states that are not initial in the given state machine.
	 * @param stateMachine the state machine
	 * @return the list of states
	 */
	public List<StateUsage> getNonInitStates(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getNonInitStates(stateMachine);
	}

	/**
	 * Returns the names of states that are not initial in the given state machine.
	 * @param stateMachine the state machine
	 * @return the names of the states
	 */
	public List<String> getNonInitStatesNameList(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getIntermediateStatesNameList(stateMachine);
	}

	/**
	 * Returns the names of initial transitions in the given state machine.
	 * @param stateMachine the state machine
	 * @return the names of the transitions
	 */
	public List<String> getInitTransitionsNameList(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getInitTransitionsNameList(stateMachine);
	}

	/**
	 * Returns the first nominal state machine in the given part usage.
	 * @param component the part usage containing the state machine
	 * @return the state machine
	 */
	public StateUsage getFirstNominalStateMachine(PartUsage component) {
		return (StateUsage) STATE_MACHINE_MODEL.getFirstNominalStateMachine(component);
	}

	/**
	 * Returns the subcomponent given its name.
	 * @param component the owner part usage
	 * @param subCompName the name of the subcomponent
	 * @return the subcomponent
	 */
	public PartUsage getSubComponent(PartUsage component, String subCompName) {
		return (PartUsage) SYSTEM_MODEL.getSubComponent(component, subCompName);
	}

	/**
	 * Returns the name of the given transition.
	 * @param transition the transition
	 * @return the name of the transition
	 */
	public String getTransitionName(Transition transition) {
		return STATE_MACHINE_MODEL.getTransitionName(transition);
	}

	/**
	 * Returns the names of the transitions of the given state machine.
	 * @param stateMachine the state machine
	 * @return the names of the transitions
	 */
	public List<String> getTransitionsNameList(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getTransitionsNameList(stateMachine);
	}

	/**
	 * Returns the name of the source state of the given transition
	 * @param transition the transition
	 * @return the name of the state
	 */
	public String getTransitionSourceStateName(Transition transition) {
		return transition.getSource().getName();
	}

	/**
	 * Returns the name of the target state of the given transition
	 * @param transition the transition
	 * @return the name of the state
	 */
	public String getTransitionTargetStateName(Transition transition) {
		return transition.getDestination().getName();
	}

	/**
	 * Returns the state invariant present in the given state.
	 * @param state the state
	 * @return the body of the state invariant
	 */
	public String getStateInvariant(StateUsage state) {
		return STATE_MACHINE_MODEL.getStateInvariant(state, null);
	}

	/**
	 * Returns the name of the given state machine.
	 * @param stateMachine the state machine
	 * @return the name of the state machine
	 */
	public String getStateMachineName(StateUsage stateMachine) {
		return stateMachine.getName();
	}

	/**
	 * Returns the owner of the given state machine.
	 * @param stateMachine the state machine
	 * @return the part definition owning the state machine
	 */
	public PartDefinition getStateMachineOwner(StateUsage stateMachine) {
		return (PartDefinition) stateMachine.eContainer();
	}

	/**
	 * Returns the name of the owner of the given state machine.
	 * @param stateMachine the state machine
	 * @return the name of the part definition owning the state machine
	 */
	public String getStateMachineOwnerName(StateUsage stateMachine) {
		return getStateMachineOwner(stateMachine).getName();
	}

	/**
	 * Returns the body of the guard of the given transition.
	 * @param transition the transition
	 * @return the body of the guard
	 */
	public String getTransitionGuardText(Transition transition) {
		return STATE_MACHINE_MODEL.getTransitionGuardCondition(transition, Language.CLEANC_STRING);
	}

	/**
	 * Returns the body of the guard of the given transition.
	 * @param transition the transition
	 * @return the body of the guard
	 */
	public String getTransitionEffectText(Transition transition) {
		return STATE_MACHINE_MODEL.getTransitionEffectText(transition, Language.CLEANC_STRING);
	}

	/**
	 * Returns the list of subcomponents of the given part usage
	 * @param component the part usage
	 * @return the list of subcomponents
	 */
	public List<PartUsage> getSubComponents(PartUsage component) {
		return SYSTEM_MODEL.getSubComponentsInstances(component);
	}

	/**
	 * Returns the names of the attributes in the given part usage or definition.
	 * @param component the component
	 * @return the names of the attributes
	 */
	public List<String> getAttributesNames(Object component) {
		return SYSTEM_MODEL.getAttributesNames(component);
	}

	/**
	 * Returns the name of the given attribute.
	 * @param attribute the attribute
	 * @return the name of the attribute
	 * @deprecated use getName(EObject) instead
	 */
	@Deprecated
	public String getAttributeName(AttributeUsage attribute) {
		return attribute.getName();
	}

	/**
	 * Returns the type of the given attribute.
	 * @param attribute the attribute
	 * @return the type of the attribute
	 * @deprecated use getType(TypedElement instead)
	 */
	@Deprecated
	public Definition getAttributeType(Usage attribute) {
		return SysMLv2Util.getType(attribute);
	}

	/**
	 * Returns the type of the given element, or null if the element is not typed.
	 * @param element the element
	 * @return the type of the element
	 */
	public Definition getType(TypedElement element) {
		return SysMLv2Util.getType(element);
	}

	/**
	 * Returns the name of the given part usage.
	 * @param component the part usage
	 * @return the name of the part usage
	 * @deprecated use getName(EObject) instead
	 */
	@Deprecated
	public String getPartUsageName(PartUsage component) {
		return component.getName();
	}

	/**
	 * Returns the name of the type of the given part usage.
	 * @param component the part usage
	 * @return the name of the type
	 * @deprecated use getComponentTypeName(Part)
	 */
	@Deprecated
	public String getPartUsageTypeName(PartUsage component) {
		return getComponentTypeName(component);
	}

	/**
	 * Returns the name of the part usage or definition.
	 * @param component the component
	 * @return the name
	 */
	public String getComponentName(Part component) {
		return getName(component);
	}

	/**
	 * Returns all the components (part definitions) of the given model.
	 * @param model the root element (Model)
	 * @return the list of part definitions
	 */
	public List<PartDefinition> getAllComponentsFromModel(Model model) {
		return SYSTEM_MODEL.getComponents(model);
	}

	/**
	 * Returns all the state machines of the given model.
	 * @param model the root element (Model)
	 * @return the set of state machines
	 */
	public Set<StateUsage> getAllStateMachinesFromModel(Model model) {
		return STATE_MACHINE_MODEL.getAllStateMachinesFromModel(model).stream().map(StateUsage.class::cast).collect(Collectors.toSet());
	}

	/**
	 * Returns the type of the part usage or definition.
	 * @param component the component
	 * @return the type
	 */
	public PartDefinition getComponentType(Part part) {
		return ModelUtil.getComponentType(part);
	}

	/**
	 * Returns the name of the type of the part usage or definition.
	 * @param part the part usage or definition
	 * @return the name of the type
	 */
	public String getComponentTypeName(Part part) {
		return getName(getComponentType(part));
	}

	/**
	 * Returns the connections between ports in the given part usage.
	 * @param component the part usage
	 * @return the list of connections (Connection or Binding)
	 */
	public List<ConnectionElement> getConnections(PartUsage component) {
		final List<ConnectionElement> allConnections = ModelUtil.getAllContentsOfTypeFromModel(
				EcoreUtil2.getResourceSet(component), ConnectionElement.class);
		allConnections.removeIf(connection -> !EcoreUtil2.equals(component, getSourceEndOwner(connection)) &&
				!EcoreUtil2.equals(component, getTargetEndOwner(connection)));
		return allConnections;
	}

	/**
	 * Return the part usage that owns the source element of the given connection.
	 * @param connection the connection
	 * @return the part usage
	 */
	public PartUsage getSourceEndOwner(ConnectionElement connection) {
		return (PartUsage) ModelUtil.getConnectorEndOwner(connection.getSource());
	}

	/**
	 * Return the part usage that owns the target element of the given connection.
	 * @param connection the connection
	 * @return the part usage
	 */
	public PartUsage getTargetEndOwner(ConnectionElement connection) {
		return (PartUsage) ModelUtil.getConnectorEndOwner(connection.getTarget());
	}

	/**
	 * Returns the source of the given connection.
	 * @param connection the Connection or the Binding
	 * @return the source element of the connection
	 */
	public NamedElement getConnectionSource(ConnectionElement connection) {
		return ((ReferenceExpression) connection.getSource()).getReferencedElement();
	}

	/**
	 * Returns the target of the given connection.
	 * @param connection the Connection or the Binding
	 * @return the target element of the connection
	 */
	public NamedElement getConnectionTarget(ConnectionElement connection) {
		return connection.getTarget().getReferencedElement();
	}

	/**
	 * Returns the name of the given element, if possible.
	 * @param element the element
	 * @return the name
	 */
	public String getName(EObject element) {
		return element instanceof final NamedElement namedElement ? namedElement.getName() : null;
	}

	/**
	 * Returns the set of state machines in the given part usage and its subcomponents.
	 * @param component the part usage
	 * @return the set of state machines
	 */
	public Set<StateUsage> getNominalStateMachinesIncludingFromSubComponents(PartUsage component) {
		return STATE_MACHINE_MODEL.getNominalStateMachinesIncludingFromSubComponents(component).stream()
			.map(StateUsage.class::cast)
			.collect(Collectors.toSet());
	}

	/**
	 * Returns the nearest containing component of the given part usage.
	 * @param component the part usage
	 * @return the nearest containing component
	 */
	public PartDefinition getNearestOwnerComponent(PartUsage component) {
		return EcoreUtil2.getContainerOfType(component, PartDefinition.class);
	}

	/**
	 * Returns the list of non-static ports of the given part usage or definition.
	 * @param component the part usage or definition
	 * @return the list of non-static ports
	 */
	public List<DirectedElement> getNonStaticPorts(EObject component) {
		return ModelUtil.getAttributeUsagesInPart(component).stream()
				.filter(DirectedElement.class::isInstance).map(DirectedElement.class::cast)
				.filter(usage -> !Direction.NONE.equals(usage.getDirection()))
				.toList();
	}

	/**
	 * Returns the name of the given port.
	 * @param port the port
	 * @return the name of the port
	 * @deprecated use getName(EObject) instead
	 */
	@Deprecated
	public String getPortName(Usage port) {
		return port.getName();
	}

	/**
	 * Returns the type of the given port.
	 * @param port the port
	 * @return the type of the port
	 * @deprecated use getAttributeType(Usage) instead
	 */
	@Deprecated
	public Definition getPortType(Usage port) {
		return SysMLv2Util.getType(port);
	}

	/**
	 * Returns the list of static ports of the given part usage or definition.
	 * Static ports do not have a direction.
	 * @param component the part usage or definition
	 * @return the list of static ports
	 */
	public List<DirectedElement> getStaticPorts(Object component) {
		return List.of();
	}

	public boolean isBooleanType(Definition type) {
		return ModelUtil.isBooleanType(type);
	}

	public boolean isClockType(Definition type) {
		return ModelUtil.isClockType(type);
	}

	public boolean isEnumType(Definition type) {
		return ModelUtil.isEnumType(type);
	}

	public boolean isIntType(Definition type) {
		return ModelUtil.isIntegerType(type);
	}

	public boolean isRealType(Definition type) {
		return ModelUtil.isRealType(type);
	}

	public boolean isIntervalType(Definition type) {
		return ModelUtil.isIntervalType(type);
	}

	public boolean isBoundedIntType(Definition type) {
		return ModelUtil.isBoundedIntType(type);
	}

	public boolean isBoundedRealType(Definition type) {
		return ModelUtil.isBoundedRealType(type);
	}

	public String[] getBounds(Container element) {
		return new String[] {
			ModelUtil.getBoundValue(element, Bound.MIN),
			ModelUtil.getBoundValue(element, Bound.MAX)
		};
	}

	/**
	 * Returns all the literals present in attributes and references of the given part definition.
	 * @param component the part definition
	 * @return the list of literals
	 */
	public List<String> getLiteralsFromAttributes(PartDefinition component) {
		return SYSTEM_MODEL.getEnumValuesFromAttributes(component);
	}

	/**
	 * Returns the list of literals in the given enumerator.
	 * @param enumerator the enumerator
	 * @return the list of literals
	 */
	public List<String> getLiteralsForEnumeratorType(EnumDefinition enumerator) {
		return ModelUtil.getEnumValues(enumerator, false);
	}

	private PartUsage getPartUsageRedefining(PartUsage instance, String redefinedInstanceName) {
		final Definition definition = ModelUtil.getComponentType(instance);
		return typeSelect(definition.getMembers(), PartUsageImpl.class).stream()
			.filter(partUsage -> partUsage.getRedefinitionName().equals(redefinedInstanceName))
			.findAny().orElseThrow(() -> new RuntimeException("No " + redefinedInstanceName + " in component instance "
					+ instance.getName() + "."));
	}

	/**
	 * Returns the part usage representing the real asset in the root part usage of the model.
	 * @param rootComponent the root part usage
	 * @return the real asset
	 */
	public PartUsage getRealAsset(PartUsage rootComponent) {
		return getPartUsageRedefining(rootComponent, REAL_ASSET);
	}

	/**
	 * Returns the part usage representing the environment (env) in the root part usage of the model.
	 * @param rootComponent the root part usage
	 * @return the environment
	 */
	public PartUsage getEnvironment(PartUsage rootComponent) {
		return getPartUsageRedefining(rootComponent, ENVIRONMENT);
	}

	private List<AttributeUsage> getAndFilterAttributes(Container container, Predicate<? super AttributeUsage> predicate) {
		final List<AttributeUsage> attributes = typeSelect(container.getMembers(), AttributeUsage.class);
		attributes.removeIf(not(predicate));
		return attributes;
	}

	/**
	 * Returns the attributes that are state variables in the given component.
	 * @param component the component
	 * @return the list of state variables
	 */
	public List<AttributeUsage> getStateVariables(PartUsage component) {
		return getAndFilterAttributes(ModelUtil.getComponentType(component), ModelUtil::isStateVariable);
	}

	/**
	 * Returns the attributes that are observable variables in the given component.
	 * @param component the component
	 * @return the list of observable variables
	 */
	public List<AttributeUsage> getObservableVariables(PartUsage component) {
		return getAndFilterAttributes(ModelUtil.getComponentType(component), SysMLv2Util::isObservableVariable);
	}

	/**
	 * Returns the observable attributes' feature chains, rooted in the system component.
	 * @param systemComponent the system component
	 * @return the list of feature chains (as strings)
	 */
	public List<String> getObservablesFeatureChains(PartUsage systemComponent) {
		return new SysMLv2FeatureChainSwitch().getObservablesFeatureChains(systemComponent);
	}

	/**
	 * Returns the attributes that are calibration parameters in the given component.
	 * @param component the component
	 * @return the list of calibration parameters
	 */
	public List<AttributeUsage> getCalibrationParameters(PartUsage component) {
		return getAndFilterAttributes(ModelUtil.getComponentType(component), ModelUtil::isCalibrationParameter);
	}

	/**
	 * Returns the default value for the given state variable.
	 * @param stateVariable the state variable
	 * @return an expression representing its default value
	 */
	public Expression getStateVariableDefault(AttributeUsage stateVariable) {
		final Value value = stateVariable.getValue();
		if (value != null && value.isDefault()) {
			return value.getExpression();
		} else {
			return null;
		}
	}

	/**
	 * Returns the part usages that represent subsystems of the real asset.
	 * @param realAsset the real asset
	 * @return the list of subsystems
	 */
	public List<PartUsage> getSubsystems(PartUsage realAsset) {
		final Definition definition = ModelUtil.getComponentType(realAsset);
		final List<PartUsage> partUsages = typeSelect(definition.getMembers(), PartUsage.class);
		partUsages.removeIf(not(ModelUtil::isSubsystem));
		return partUsages;
	}

	/**
	 * Returns whether the attribute represents an observable variable.
	 * @param variable the attribute
	 * @return a boolean indicating whether the attribute represents an observable variable
	 */
	public boolean isObservable(AttributeUsage variable) {
		return SysMLv2Util.isObservableVariable(variable);
	}

	/**
	 * Returns whether the activity is unplannable.
	 * @param activity the activity
	 * @return a boolean indicating whether the activity is unplannable
	 */
	public boolean isUnplannable(PartUsage activity) {
		final Definition definition = ModelUtil.getComponentType(activity);
		return SysMLv2Util.containsMetadataUsagesOfType(definition, UNPLANNABLE);
	}

	/**
	 * Returns the part usages that represent activities of the given subsystem.
	 * @param subsystem the subsystem
	 * @return the list of activities
	 */
	public List<PartUsage> getActivities(PartUsage subsystem) {
		final Definition definition = ModelUtil.getComponentType(subsystem);
		final List<PartUsage> partUsages = typeSelect(definition.getMembers(), PartUsage.class);
		partUsages.removeIf(not(ModelUtil::isActivity));
		return partUsages;
	}

	/**
	 * Returns the list of activity parameters.
	 * @param activity the activity
	 * @return a list of its parameters
	 */
	public List<Usage> getActivityParameters(PartUsage activity) {
		final Definition definition = ModelUtil.getComponentType(activity);
		final List<Usage> usages = typeSelect(definition.getMembers(), Usage.class);
		usages.removeIf(usage -> !SysMLv2Util.containsMetadataUsagesOfType(usage, ACTIVITY_PARAM));
		return usages;
	}

	/**
	 * Returns the subsystem's platform state machine.
	 * @param subsystem the subsystem
	 * @return the platform state machine
	 */
	public StateUsage getPlatformStateMachine(PartUsage subsystem) {
		return ModelUtil.getPlatformStateMachine(subsystem);
	}

	/**
	 * Returns the subsystem's mission state machine.
	 * @param subsystem the subsystem
	 * @return the mission state machine
	 */
	public StateUsage getMissionStateMachine(PartUsage subsystem) {
		return ModelUtil.getMissionStateMachine(subsystem);
	}

	/**
	 * Returns the list of durative transitions present in the given state machine.
	 * @param stateMachine the state machine
	 * @return the list of durative transitions
	 */
	public List<Transition> getDurativeTransitions(StateUsage stateMachine) {
		final List<Transition> transitions = getTransitions(stateMachine);
		transitions.removeIf(not(SysMLv2Util::isDurative));
		return transitions;
	}

	/**
	 * Returns the list of all the transitions in the given state machine.
	 * @param stateMachine the state machine
	 * @return the list of all transitions
	 */
	public List<Transition> getTransitions(StateUsage stateMachine) {
		return STATE_MACHINE_MODEL.getTransitions(stateMachine);
	}

	/**
	 * Returns the source state of the given transition.
	 * @param transition the transition
	 * @return the source state
	 */
	public StateUsage getTransitionSource(Transition transition) {
		return transition.getSource();
	}

	/**
	 * Returns the destination state of the given transition.
	 * @param transition the transition
	 * @return the destination state
	 */
	public StateUsage getTransitionDestination(Transition transition) {
		return transition.getDestination();
	}

	/**
	 * Returns the activity associated to the given durative transition.
	 * @param durativeTransition the durative transition
	 * @return the associated activity
	 */
	public PartUsage getTransitionActivity(Transition durativeTransition) {
		final EObject act = durativeTransition.getMembers().get(0);
		if (act instanceof final Feature feature) {
			Expression expression = feature.getValue().getExpression();
			if (expression instanceof final ReferenceExpression referenceExpression) {
				final NamedElement called = referenceExpression.getReferencedElement();
				if (called instanceof final PartUsage partUsage) {
					return partUsage;
				}
			}
		}
		throw new RuntimeException("Ill-formed DurativeTransition " + durativeTransition.getName());
	}

	/**
	 * Returns the precondition of the given activity.
	 * @param activity the activity
	 * @return the precondition
	 */
	public ConstraintUsage getPreCondition(PartUsage activity) {
		return SysMLv2Util.getMemberByName(ModelUtil.getComponentType(activity), ConstraintUsage.class, PRECONDITION);
	}

	/**
	 * Returns the postcondition of the given activity.
	 * @param activity the activity
	 * @return the postcondition
	 */
	public ConstraintUsage getPostCondition(PartUsage activity) {
		return SysMLv2Util.getMemberByName(ModelUtil.getComponentType(activity), ConstraintUsage.class, POSTCONDITION);
	}

	/**
	 * Returns the preeffect of the given activity.
	 * @param activity the activity
	 * @return the preeffect
	 */
	public ActionUsage getPreEffect(PartUsage activity) {
		return SysMLv2Util.getMemberByName(ModelUtil.getComponentType(activity), ActionUsage.class, PREEFFECT);
	}

	/**
	 * Returns the posteffect of the given activity.
	 * @param activity the activity
	 * @return the posteffect
	 */
	public ActionUsage getPostEffect(PartUsage activity) {
		return SysMLv2Util.getMemberByName(ModelUtil.getComponentType(activity), ActionUsage.class, POSTEFFECT);
	}

	/**
	 * Returns all members inside a pre/post effect
	 * @param effect the pre/post effect
	 * @return the list of objects representing the members of the effect
	 * @since 2025-04-23
	 */
	public List<EObject> getEffectMembers(ActionUsage effect) {
		return effect.getMembers();
	}

	/**
	 * Returns true if the passed object is an assignment
	 * @param member the object
	 * @return true if it is an assignment, false otherwise
	 * @since 2025-04-23
	 */
	public boolean isAssignment(EObject member) {
		return member instanceof Assignment;
	}

	/**
	 * Returns true if the passed object is a 'send' statement representing a Setter action
	 * @param member the object
	 * @return true if it is a 'send' representing a Setter, false otherwise
	 * @since 2025-04-23
	 */
	public boolean isSetter(EObject member) {
		return member instanceof final SendUsage sendUsage && sendUsage.getTransport().getReferencedElement().getName().startsWith("set_");
	}

	/**
	 * Returns the pair of variable and value set by the given Assignment
	 * @param assignment the assignment
	 * @return an ordered pair of objects representing the variable and the value
	 * @since 2025-04-23
	 */
	public Pair<NamedElement, Expression> getVariableAndValue(Assignment assignment) {
		return new Pair<NamedElement, Expression>(assignment.getElement().getReferencedElement(),
				assignment.getValue().getExpression());
	}

	/**
	 * Returns the pair of variable and value set by the given Setter action
	 * @param sendUsage the 'send' statement representing the Setter action
	 * @return an ordered pair of objects representing the variable name and the value
	 * @since 2025-04-23
	 */
	public Pair<String, Expression> getVariableAndValue(SendUsage sendUsage) {
		return new Pair<String, Expression>(sendUsage.getTransport().getReferencedElement().getName().substring(4),
				sendUsage.getPayload());
	}

	/**
	 * Returns the invariant of the given activity.
	 * @param activity the activity
	 * @return the invariant
	 */
	public ConstraintUsage getInvariant(PartUsage activity) {
		return SysMLv2Util.getMemberByName(ModelUtil.getComponentType(activity), ConstraintUsage.class, INVARIANT);
	}

	private Expression getExpression(Calculation calculation) {
		if (calculation != null) {
			if (calculation.getResult() instanceof final Expression expression) {
				return expression;
			}
		}
		return null;
	}

	/**
	 * Returns the lower bound for the activity duration.
	 * @param activity the activity
	 * @return an expression representing the lower bound for its duration
	 */
	public Expression getLowerDuration(PartUsage activity) {
		final CalculationUsage calculation = SysMLv2Util.getMemberByName(
				ModelUtil.getComponentType(activity), CalculationUsage.class, GET_LOWER_BOUND_DURATION);
		return getExpression(calculation);
	}

	/**
	 * Returns the upper bound for the activity duration.
	 * @param activity the activity
	 * @return an expression representing the upper bound for its duration
	 */
	public Object getUpperDuration(PartUsage activity) {
		final CalculationUsage calculation = SysMLv2Util.getMemberByName(
				ModelUtil.getComponentType(activity), CalculationUsage.class, GET_UPPER_BOUND_DURATION);
		return getExpression(calculation);
	}

	/**
	 * Returns a list of (variable, value) pairs representing all the assignments in an Action.
	 * @param action the action
	 * @return a list of (variable, value) pairs. To obtain the variable and the expression representing
	 * the value, call getLeft() and getRight() on each pair.
	 */
	public List<Pair<NamedElement, Expression>> getAssignments(ActionUsage action) {
		final List<Assignment> assignments = typeSelect(action.getMembers(), Assignment.class);
		final List<Pair<NamedElement, Expression>> pairs = new ArrayList<>();
		assignments.forEach(assignment -> {
			final Pair<NamedElement, Expression> pair = new Pair<NamedElement, Expression>(assignment.getElement().getReferencedElement(),
					assignment.getValue().getExpression());
			pairs.add(pair);
		});
		return pairs;
	}

	/**
	 * Returns the string corresponding to the given expression in the SysML v2 model.
	 * @param expression the expression
	 * @return the original string in the model.
	 */
	public String serialize(Expression expression) {
		return NodeModelUtils.findActualNodeFor(expression).getText();
	}

	/**
	 * Translates the given expression in the SysML v2 model into its SMV equivalent expression.
	 * @param expression the expression
	 * @return its SMV equivalent.
	 */
	public String serializeSMVCompatible(Expression expression) {
		return SysMLv2ToSMVExpression.getLTLSMVInstanceWithoutEnumMapping().serialize(expression);
	}

	/**
	 * Returns the list of uninterpreted functions invoked by this component.
	 * @param component the component (i.e. a subsystem or activity)
	 * @return the list of uninterpreted functions invoked by this component
	 */
	public List<Calculation> getUninterpretedFunctions(PartUsage component) {
		final List<Calculation> functions = SYSTEM_MODEL.getUninterpretedFunctions(component);
		getActivities(component).forEach(activity -> functions.addAll(getUninterpretedFunctions(activity)));
		return functions;
	}

	/**
	 * Returns the list of interpreted functions invoked by this component.
	 * @param component the component (i.e. a subsystem or activity)
	 * @return the list of interpreted functions invoked by this component
	 */
	public List<Calculation> getInterpretedFunctions(PartUsage component) {
		final Definition partDefinition = ModelUtil.getComponentType(component);
		final List<Calculation> functions = EcoreUtil2.getAllContentsOfType(partDefinition,
				ReferenceExpression.class).stream()
			.map(ReferenceExpression::getReferencedElement)
			.filter(Calculation.class::isInstance).map(Calculation.class::cast)
			.filter(ModelUtil::isInterpretedFunction)
			.collect(Collectors.toList());
		getActivities(component).forEach(activity -> functions.addAll(getInterpretedFunctions(activity)));
		return functions;
	}

	/**
	 * Returns the list of the names of the inputs of the given uninterpreted function.
	 * @param uninterpretedFunction the uninterpreted function
	 * @return the list of the names of its input parameters
	 */
	public List<String> getUninterpretedFunctionInputNames(Object uninterpretedFunction) {
		return SYSTEM_MODEL.getUninterpretedFunctionInputs(uninterpretedFunction)
			.map(this::getName)
			.toList();
	}

	/**
	 * Returns the list of the types of the inputs of the given uninterpreted function.
	 * @param uninterpretedFunction the uninterpreted function
	 * @return the list of the types of its input parameters
	 */
	public List<Definition> getUninterpretedFunctionInputTypes(Calculation uninterpretedFunction) {
		return SYSTEM_MODEL.getUninterpretedFunctionInputs(uninterpretedFunction)
				.map(ModelUtil::getUnboundedType)
				.toList();
	}

	/**
	 * Returns the list of the multiplicities of the inputs of the given uninterpreted function.
	 * @param uninterpretedFunction the uninterpreted function
	 * @return the list of the multiplicities of its input parameters
	 */
	public List<String[]> getUninterpretedFunctionInputMultiplicities(Calculation uninterpretedFunction) {
		return SYSTEM_MODEL.getUninterpretedFunctionInputMultiplicities(uninterpretedFunction);
	}

	/**
	 * Returns the type of the object returned by given uninterpreted function.
	 * @param uninterpretedFunction the uninterpreted function
	 * @return the type of its returned object
	 */
	public Definition getUninterpretedFunctionOutputType(Calculation uninterpretedFunction) {
		return SYSTEM_MODEL.getUninterpretedFunctionOutputType(uninterpretedFunction);
	}

	/**
	 * Returns the multiplicity of the object returned by the given uninterpreted function.
	 * @param uninterpretedFunction the uninterpreted function
	 * @return the multiplicity of its returned object
	 */
	public String[] getUninterpretedFunctionOutputMultiplicity(Calculation uninterpretedFunction) {
		return SYSTEM_MODEL.getUninterpretedFunctionOutputMultiplicity(uninterpretedFunction);
	}
}
