/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.api.core;

import static com.google.common.base.Predicates.or;
import static java.util.function.Predicate.not;
import static org.eclipse.emf.common.util.ECollections.emptyEList;
import static org.eclipse.emf.common.util.ECollections.toEList;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.emf.common.util.ECollections;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.resource.XtextResource;

import eu.fbk.eclipse.explodtwin.api.exception.NotAPartException;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression;
import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression.Language;
import eu.fbk.eclipse.standardtools.utils.core.model.AbstractStateMachineModelClass;
import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.sysmlv2.sysMLv2.ActionUsage;
import eu.fbk.sysmlv2.sysMLv2.AssertionAction;
import eu.fbk.sysmlv2.sysMLv2.Assignment;
import eu.fbk.sysmlv2.sysMLv2.Calculation;
import eu.fbk.sysmlv2.sysMLv2.Definition;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Direction;
import eu.fbk.sysmlv2.sysMLv2.Effect;
import eu.fbk.sysmlv2.sysMLv2.EntryAction;
import eu.fbk.sysmlv2.sysMLv2.EntryTransition;
import eu.fbk.sysmlv2.sysMLv2.EnumDefinition;
import eu.fbk.sysmlv2.sysMLv2.Expression;
import eu.fbk.sysmlv2.sysMLv2.Guard;
import eu.fbk.sysmlv2.sysMLv2.ItemDefinition;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.Part;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.PerformActionReference;
import eu.fbk.sysmlv2.sysMLv2.PortUsage;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.SendUsage;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.SysMLv2Factory;
import eu.fbk.sysmlv2.sysMLv2.TernaryOperatorExpression;
import eu.fbk.sysmlv2.sysMLv2.Transition;
import eu.fbk.sysmlv2.sysMLv2.Usage;
import eu.fbk.sysmlv2.sysMLv2.Value;
import eu.fbk.sysmlv2.sysMLv2.impl.UsageImplCustom;
import eu.fbk.sysmlv2.util.SysMLv2Util;

public class SysMLv2StateMachineModel extends AbstractStateMachineModelClass {

	@Override
	public Set<StateUsage> getAllStateMachinesFromModel(Object model) {
		if (model instanceof final XtextResource xtextResource) {
			final EList<EObject> rootContents = xtextResource.getContents();
			if (rootContents.isEmpty()) {
				return null;
			}
			final List<StateUsage> stateUsageList = EcoreUtil2.getAllContentsOfType(rootContents.get(0), StateUsage.class);
			if (stateUsageList.isEmpty()) {
				return null;
			}
			return stateUsageList.stream()
				.filter(SysMLv2Util::isStateMachine)
				.filter(not(ModelUtil::isNominalStateMachine))
				.collect(Collectors.toSet());
		} else {
			return null;
		}
	}

	@Override
	public String getStateMachineName(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			return stateUsage.getName();
		}
		return null;
	}

	@Override
	public Object getFirstNominalStateMachine(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		return ModelUtil.getNominalStateMachinesOfComponent(part).stream().findFirst().orElse(null);
	}

	@Override
	public Object getStateMachineOwner(Object stateMachine) {
		if (!(stateMachine instanceof final StateUsage stateUsage)) {
			throw new IllegalArgumentException("Argument is not a state machine.");
		}
		return EcoreUtil2.getContainerOfType(stateUsage, PartDefinition.class);
	}

	@Override
	public String getStateMachineOwnerName(Object stateMachine) {
		final Object owner = getStateMachineOwner(stateMachine);
		return owner instanceof final NamedElement namedElement ? namedElement.getName() : null;
	}

	@Override
	public EList<String> getIntermediateStatesNameList(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			return typeSelect(stateUsage.getMembers(), StateUsage.class).stream()
				.map(StateUsage::getName)
				.collect(Collectors.toCollection(ECollections::newBasicEList));
		} else {
			return null;
		}
	}

	@Override
	public EList<String> getTransitionsNameList(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			return getTransitions(stateUsage).stream()
				.map(this::getTransitionName)
				.collect(Collectors.toCollection(ECollections::newBasicEList));
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public EList<Transition> getInitTransitions(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			final EList<Transition> transitions = getTransitions(stateUsage);
			transitions.removeIf(not(EntryTransition.class::isInstance));
			if (transitions.isEmpty()) {
				throw new RuntimeException("No entry transitions in state machine " + stateUsage.getName());
			} else {
				return transitions;
			}
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public EList<Transition> getNonInitTransitions(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			final EList<Transition> transitions = getTransitions(stateUsage);
			transitions.removeIf(EntryTransition.class::isInstance);
			if (transitions.isEmpty()) {
				throw new RuntimeException("No transitions in state machine " + stateUsage.getName());
			} else {
				return transitions;
			}
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public EList<Transition> getTransitions(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			return toEList(typeSelect(stateUsage.getMembers(), Transition.class));
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public EList<String> getInitTransitionsNameList(Object stateMachine) {
		return getInitTransitions(stateMachine).stream()
			.map(this::getTransitionName)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public EList<String> getNonInitTransitionsNameList(Object stateMachine) {
		return getNonInitTransitions(stateMachine).stream()
				.map(this::getTransitionName)
				.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public EList<StateUsage> getNonInitStates(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			final EList<StateUsage> states = toEList(typeSelect(stateUsage.getMembers(), StateUsage.class));
			if (states.isEmpty()) {
				throw new RuntimeException("No states in state machine " + stateUsage.getName());
			}
			return states;
		} else {
			throw new RuntimeException("Unexpected argument type.");
		}
	}

	private final Map<Pair<String, String>, Integer> disambiguationCounterMap = new HashMap<>();
	private final Map<Transition, String> transitionNamesMap = new HashMap<>();
	@Override
	public String getTransitionName(Object element) {
		if (element instanceof final Transition transition) {
			if (transition.getName() == null) {
				if (transitionNamesMap.get(transition) == null) {
					final String sourceStateName = transition.getSource().getName();
					final String destinationStateName = transition.getDestination().getName();
					String name = "_" + sourceStateName + "_to_" + destinationStateName + "_";
					final String componentName = EcoreUtil2.getContainerOfType(transition, PartDefinition.class).getName();
					final var key = new Pair<>(componentName, name);
					Integer counter = disambiguationCounterMap.get(key);
					if (counter == null) {
						disambiguationCounterMap.put(key, 0);
					} else {
						disambiguationCounterMap.put(key, ++counter);
						name += counter + "_";
					}
					transitionNamesMap.put(transition, name);
					return name;
				} else {
					return transitionNamesMap.get(transition);
				}
			} else {
				final String originalName = transition.getName();
				if (originalName.startsWith("'")) {
					final String unquotedName = originalName.substring(1, originalName.length() - 1);
					unquotedName.replace("\\s+", "_");
					return unquotedName;
				}
				return originalName;
			}
		}
		return null;
	}

	@Override
	public String getTransitionNextStateName(Object element) {
		if (element instanceof final Transition transition) {
			return transition.getDestination().getName();
		}
		return null;
	}

	@Override
	public String getTransitionStartStateName(Object element) {
		if (element instanceof final Transition transition) {
			return transition.getSource().getName();
		}
		return null;
	}

	@Override
	public String getTransitionGuardCondition(Object element, String language) {
		if (element instanceof final Transition transition) {
			final StringBuilder serializedCondition = new StringBuilder();
			final var flattenedEventsNames = transition.getTriggers().stream()
				.map(trigger -> {
					final PortUsage portUsage = (PortUsage) trigger.getPort().getReferencedElement();
					final Definition portDefinition = SysMLv2Util.getType(portUsage);
					final Definition payload = SysMLv2Util.getType(trigger.getPayload());
					final DirectedElement eventMember = typeSelect(portDefinition.getMembers(), DirectedElement.class).stream()
						.filter(member -> SysMLv2Util.isEvent(member) &&
							SysMLv2Util.getType(member).getName().equals(payload.getName()))
						.findAny().orElseThrow(() ->
							new RuntimeException("Port usage " + portUsage.getName() + " is typed by"
									+ " port definition " + portDefinition.getName() + ", that does not contain"
									+ " a member of type " + payload.getName()));
					final NamedElement flattenedPort = ModelUtil.flattenNestedMember(eventMember, portUsage);
					return flattenedPort;
				})
				.map(NamedElement::getName)
				.toList();

			final Map<String, List<String>> groupedByPrefix = new HashMap<>();
			for (final String str : flattenedEventsNames) {
				final int underscoreIndex = str.indexOf('_');
				final String prefix = str.substring(0, underscoreIndex);
				groupedByPrefix
					.computeIfAbsent(prefix, k -> new ArrayList<>())
					.add(str);
	        }

			if (groupedByPrefix.containsKey("performed")) {
				final var list = groupedByPrefix.get("performed");
				final var listSize = list.size();
				serializedCondition.append("(");
				for (int i = 0; i < listSize; i++) {
					final String string = list.get(i);
					serializedCondition.append(string).append(" == true");
					if (i != listSize - 1) {
						serializedCondition.append(" && ");
					}
				}
				serializedCondition.append(")");
			}

			if (groupedByPrefix.containsKey("startEvent")) {
				if (!serializedCondition.isEmpty()) {
					serializedCondition.append(" && ");
				}
				final var list = groupedByPrefix.get("startEvent");
				final var listSize = list.size();
				serializedCondition.append("(");
				for (int i = 0; i < listSize; i++) {
					final String string = list.get(i);
					serializedCondition.append(string).append(" == true");
					if (i != listSize - 1) {
						serializedCondition.append(" || ");
					}
				}
				serializedCondition.append(")");
			}

			final List<String> misc = groupedByPrefix.entrySet().stream()
				.filter(entry -> !(entry.getKey().equals("performed") || entry.getKey().equals("startEvent")))
				.map(Entry::getValue)
				.flatMap(List::stream)
				.toList();
			if (!misc.isEmpty()) {
				if (!serializedCondition.isEmpty()) {
					serializedCondition.append(" && ");
				}
				serializedCondition.append("(");
				serializedCondition.append(misc.stream().collect(Collectors.joining(" || ")));
				serializedCondition.append(")");
			}

			final Guard guard = transition.getGuard();
			if (guard != null) {
				if (!serializedCondition.isEmpty()) {
					serializedCondition.append(" && ");
				}
				serializedCondition.append(SysMLv2ToSMVExpression.getInstance(language).serialize(guard.getExpression()));
			}
			// Compute the !(event1 || event2 || ... || eventN) guard for stutter self-loops
			if (transition.isStutter()) {
				final var stateMachine = EcoreUtil2.getContainerOfType(transition, StateUsage.class);
				final var isMission = stateMachine.getSubsetting() != null
						&& stateMachine.getSubsetting().getReferences().get(0).getReferencedElement().getName().equals("missionStateMachines");
				final String eventsGuard = getOwnerInputEvents(stateMachine).stream()
					.map(NamedElement::getName)
					.filter(name -> !isMission || !name.startsWith("startEvent"))
					.collect(Collectors.joining(" || "));
				serializedCondition.append("!(").append(eventsGuard).append(")");
			}
			if (!serializedCondition.isEmpty()) {
				return serializedCondition.toString();
			}
		}
		return null;
	}

	private String translateActionElement(EObject element, String language) {
		final StringBuilder text = new StringBuilder();
		if (element instanceof final Assignment assignmentInAction) {
			text.append(extractAssignment(assignmentInAction, language));
		} else if (element instanceof final SendUsage sendUsageInAction) {
			text.append(translateSendUsage(sendUsageInAction));
		} else if (element instanceof final PerformActionReference performInAction) {
			final var performedAction = performInAction.getReference().getReferencedElement();
			if (performedAction instanceof final ActionUsage actionUsage) {
				text.append(translateActionElement(actionUsage, language));
			}
		} else if (element instanceof final ActionUsage actionUsage) {
			actionUsage.getMembers().stream()
				.filter(member -> member instanceof Assignment
						|| member instanceof SendUsage
						|| member instanceof PerformActionReference)
				.forEach(member -> text.append(translateActionElement(member, language)));
		} else {
			throw new RuntimeException("Unexpected argument type.");
		}
		return text.toString();
	}

	private String translateEffect(Effect effect, String language) {
		return translateActionElement(effect.getAction(), language);
	}

	@Override
	public String getTransitionEffectText(Object object, String language) {
		final Effect effect;
		final StateUsage destinationState;
		if (object instanceof final EntryTransition entryTransition) {
			// In practice, there will be one entry action per state machine
			final var entryActions = EcoreUtil2.getSiblingsOfType(entryTransition, EntryAction.class);
			if (!entryActions.isEmpty()) {
				effect = entryActions.get(0).getEffect();
			} else {
				effect = null;
			}
			destinationState = entryTransition.getDestination();
		} else if (object instanceof final Transition transition) {
			effect = transition.getEffect();
			destinationState = transition.getDestination();
		} else {
			throw new IllegalArgumentException("Unexpected argument type.");
		}
		final StringBuilder effectText = new StringBuilder();
		if (effect != null) {
			effectText.append(translateEffect(effect, language));
		}
		final var destinationEntryActions = typeSelect(destinationState.getMembers(), EntryAction.class);
		if (!destinationEntryActions.isEmpty()) {
			// In practice, there will be at most one entry action per state.
			effectText.append(translateEffect(destinationEntryActions.get(0).getEffect(), language));
		}
		return effectText.toString();
	}

	private String translateSendUsage(SendUsage sendUsage) {
		if (sendUsage.getPayload() instanceof final ReferenceExpression payloadAsReference &&
				payloadAsReference.getReferencedElement() instanceof final ItemDefinition payload) {
			final PortUsage portUsage = (PortUsage) sendUsage.getTransport().getReferencedElement(); // TODO uniform feature names
			final Definition portDefinition = SysMLv2Util.getType(portUsage);
			final DirectedElement eventMemberToFlatten = SysMLv2Util.getEventMemberFromPort(portUsage);
			if (!SysMLv2Util.getType(eventMemberToFlatten).getName().equals(payload.getName())) {
				throw new RuntimeException("Port usage " + portUsage.getName() + " is typed by"
						+ " port definition " + portDefinition.getName() + ", that does not contain"
								+ " a member of type " + payload.getName());
			}
			final NamedElement flattenedPort = ModelUtil.flattenNestedMember(eventMemberToFlatten, portUsage);
			return flattenedPort.getName() + " = ! ( " + flattenedPort.getName() + " ) ;";
		} else {
			final Expression expression = sendUsage.getPayload();
			final var serializer = SysMLv2ToSMVExpression.getInstance(Language.CLEANC);
			if (!(sendUsage.getTransport().getReferencedElement() instanceof final PortUsage port)) {
				throw new RuntimeException("Invalid SendUsage sending " + serializer.serialize(expression)
						+ " in component " + EcoreUtil2.getContainerOfType(sendUsage, PartDefinition.class).getName()
						+ ". The transport should be a PortUsage, instead it is " + sendUsage.getTransport().getReferencedElement());
			}
			final DirectedElement dataPort = SysMLv2Util.getDataMemberFromPort(port);
			final DirectedElement flattenedDataPort = ModelUtil.flattenNestedMember(dataPort, port);
			final DirectedElement eventPort = SysMLv2Util.getEventMemberFromPort(port);
			final DirectedElement flattenedEventPort = ModelUtil.flattenNestedMember(eventPort, port);
			final Assignment dataAssignment = SysMLv2Factory.eINSTANCE.createAssignment();
			final ReferenceExpression elementReference = SysMLv2Factory.eINSTANCE.createReferenceExpression();
			elementReference.setReferencedElement(flattenedDataPort);
			dataAssignment.setElement(elementReference);
			final Value value = SysMLv2Factory.eINSTANCE.createValue();
			value.setExpression(EcoreUtil.copy(expression));
			dataAssignment.setValue(value);
			return extractAssignment(dataAssignment, Language.CLEANC_STRING)
					+ flattenedEventPort.getName() + " = ! ( " + flattenedEventPort.getName() + " ) ; ";
		}
	}

	private String extractAssignment(Assignment assignment, String language) {
		if (assignment.getValue().getExpression() instanceof final ReferenceExpression reference &&
				reference.getReferencedElement() instanceof final Calculation calculation &&
				calculation.isAbstract()) {
			return "";
		}
		final var serializer = SysMLv2ToSMVExpression.getInstance(language);
		final String referenceUsageName = serializer.serialize(assignment.getElement());
		if (assignment.getValue().getExpression() instanceof final TernaryOperatorExpression ternaryOperatorExpression) {
			final StringBuilder complexAssignment = new StringBuilder();
			final var guardExpression = ternaryOperatorExpression.getGuard().getExpression();
			final var trueExpression = ternaryOperatorExpression.getLeft();
			final var falseExpression = ternaryOperatorExpression.getRight();
			complexAssignment.append("if (")
				.append(serializer.serialize(guardExpression))
				.append(") {")
				.append(referenceUsageName).append(" = ").append(serializer.serialize(trueExpression)).append(";")
				.append("} else {")
				.append(referenceUsageName).append(" = ").append(serializer.serialize(falseExpression)).append(";")
				.append("}");
			 return complexAssignment.toString();
		} else {
			final String value = serializer.serialize(assignment.getValue().getExpression());
			return referenceUsageName + "=" + value + ";";
		}
	}

	@Override
	public String getStateInvariant(Object state, String language) {
		if (state instanceof final StateUsage stateUsage) {
			final List<AssertionAction> assertions = typeSelect(stateUsage.getMembers(), AssertionAction.class);
			return switch (assertions.size()) {
			case 0 -> null;
			case 1 -> assertionToString(assertions.get(0), language);
			default -> throw new RuntimeException("Too many state invariants for state: " + stateUsage.getName());
			};
		} else {
			return null;
		}
	}

	private String assertionToString(AssertionAction assertionAction, String language) {
		Expression expression;
		if (assertionAction.getConstraint() != null) {
			expression = assertionAction.getConstraint().getExpression();
		} else {
			expression = assertionAction.getInlineConstraint().getExpression();
		}
		return SysMLv2ToSMVExpression.getInstance(language).serialize(expression);
	}

	@Override
	public Set<StateUsage> getNominalStateMachinesIncludingFromSubComponents(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final Set<StateUsage> stateMachines = ModelUtil.getNominalStateMachinesOfComponent(part);
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		typeSelect(partDefinition.getMembers(), PartUsage.class)
			.forEach(subComponent -> stateMachines.addAll(getNominalStateMachinesIncludingFromSubComponents(subComponent)));
		return stateMachines;
	}

	@Override
	public boolean isInputPort(Object port) {
		return ModelUtil.isInputParameter(port);
	}

	@Override
	public EList<DirectedElement> getOwnerNonStaticAttributesExceptPorts(Object stateMachine) {
		final PartDefinition owner = (PartDefinition) getStateMachineOwner(stateMachine);
		final EList<Usage> attributeUsages = ModelUtil.getAttributeUsagesInPart(owner);
		return typeSelect(attributeUsages, DirectedElement.class).stream()
			.filter(not(or(ModelUtil::isOfContractType, ModelUtil::isInterfaceAssertion, ModelUtil::isOfClockType, ModelUtil::isInvariant)))
			.filter(directedElement -> directedElement.getDirection().equals(Direction.NONE))
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	//TODO: to be implemented after deciding how static ports are marked in SysMLv2
	@Override
	public EList<?> getOwnerStaticAttributesExceptPorts(Object stateMachine) {
		return ModelUtil.getDefines((PartDefinition) getStateMachineOwner(stateMachine));
	}

	//TODO: to be implemented after deciding how static ports are marked in SysMLv2
	@Override
	public EList<?> getOwnerStaticAttributes(Object stateMachine) {
		return emptyEList();
	}

	@Override
	public boolean isIdealClockType(Object type) {
		return ModelUtil.isClockType(type);
	}

	@Override
	public EList<DirectedElement> getOwnerOutputPortsExceptEvents(Object stateMachine) {
		return getOwnerNonStaticPortsFiltering(stateMachine, ModelUtil::isOutputParameter, not(SysMLv2Util::isEvent));
	}

	@Override
	public EList<DirectedElement> getOwnerInputPortsExceptEvents(Object stateMachine) {
		return getOwnerNonStaticPortsFiltering(stateMachine, ModelUtil::isInputParameter, not(SysMLv2Util::isEvent), not(ModelUtil::isHavocGeneratedAttribute), not(param -> SysMLv2Util.containsMetadataUsagesOfType(param, "ActivityParam")));
	}

	@Override
	public EList<DirectedElement> getOwnerInputEvents(Object stateMachine) {
		return getOwnerNonStaticPortsFiltering(stateMachine, ModelUtil::isInputParameter, SysMLv2Util::isEvent);
	}

	@Override
	public EList<DirectedElement> getOwnerOutputEvents(Object stateMachine) {
		return getOwnerNonStaticPortsFiltering(stateMachine, ModelUtil::isOutputParameter, SysMLv2Util::isEvent);
	}

	/*
	 * No elements are inserted into the varargs parameters array,
	 * nor is a reference to it ever exposed by this method, thus
	 * I am annotating it as @SafeVarargs.
	 */
	@SafeVarargs
	private EList<DirectedElement> getOwnerNonStaticPortsFiltering(Object stateMachineOrComponent, Predicate<? super DirectedElement>... predicates) {
		PartDefinition owner = null;
		if (stateMachineOrComponent instanceof StateUsage stateMachine) {
			owner = (PartDefinition) getStateMachineOwner(stateMachine);
		} else if (stateMachineOrComponent instanceof final PartDefinition partDefinition) {
			owner = partDefinition;
		}
		Stream<DirectedElement> portsStream = ModelUtil.getNonStaticPorts(owner).stream();
		for (final var predicate : predicates) {
			portsStream = portsStream.filter(predicate);
		}
		return portsStream.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public Object getAttributeType(Object property) {
		if (property instanceof final Usage usage) {
			return ModelUtil.getUnboundedType(usage);
		} else {
			return null;
		}
	}

	@Override
	public boolean isBooleanType(Object type) {
		return ModelUtil.isBooleanType(type);
	}

	@Override
	public boolean isIntegerType(Object type) {
		return ModelUtil.isIntegerType(type);
	}

	@Override
	public boolean isRealType(Object type) {
		return ModelUtil.isRealType(type);
	}

	/*
	 * The default implementation of this method already returns false.
	 * This overriding is not a mistake, it is just here for clarity.
	 * This class is only used for translation into SMV, and for this
	 * purpose, we map both BoundedInt and BoundedReal (which are the
	 * only two bounded/interval/range types we support) into Integer and
	 * Real, respectively, plus invariants. Thus, we indeed want no type
	 * to be considered as an interval type during translation.
	 */
	@Override
	public boolean isIntervalType(Object type) {
		return false;
	}

	@Override
	public boolean isEnumType(Object type) {
		return ModelUtil.isEnumType(type);
	}

	@Override
	public String getEnumTypeName(Object type) {
		return getAttributeName(type);
	}

	@Override
	public EList<String> getEnumValues(Object enumType) {
		return ModelUtil.getEnumValues((EnumDefinition) enumType, true);
	}

	@Override
	public String getAttributeName(Object property) {
		return property instanceof final NamedElement namedElement ? namedElement.getName() : null;
	}

	@Override
	public EList<StateUsage> getErrorStates(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage) {
			return typeSelect(stateUsage.getMembers(), StateUsage.class).stream()
				.filter(ModelUtil::isErrorState)
				.collect(Collectors.toCollection(ECollections::newBasicEList));
		} else {
			return emptyEList();
		}
	}

	@Override
	public EList<StateUsage> getFaultyStateMachines(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		return typeSelect(partDefinition.getMembers(), StateUsage.class).stream()
			.filter(not(ModelUtil::isNominalStateMachine))
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public String getErrorStateProbability(Object stateMachine) {
		if (stateMachine instanceof final StateUsage stateUsage && ModelUtil.isErrorState(stateUsage)) {
			final Usage probabilityUsage = typeSelect(stateUsage.getMembers(), UsageImplCustom.class).stream()
				.filter(usage -> "probability".equals(usage.getRedefinitionName()))
				.findFirst().orElse(null);
			if (probabilityUsage != null && probabilityUsage.getValue() != null) {
				return SysMLv2ToSMVExpression.getInstance(Language.LTL).serialize(probabilityUsage.getValue().getExpression());
			}
		}
		return null;
	}

	@Override
	public EList<Transition> getIncomingFaultTransitions(Object errorState) {
		if (errorState instanceof final StateUsage stateUsage && ModelUtil.isErrorState(stateUsage)) {
			final StateUsage stateMachine = EcoreUtil2.getContainerOfType(stateUsage.eContainer(), StateUsage.class);
			return getTransitions(stateMachine).stream()
				.filter(transition -> EcoreUtil.equals(transition.getDestination(), stateUsage))
				.collect(Collectors.toCollection(ECollections::newBasicEList));
		} else {
			return emptyEList();
		}
	}

	@Override
	public String getFailureModeType(Object errorState) {
		if (errorState instanceof final StateUsage stateUsage && stateUsage.getTyping() != null) {
			return SysMLv2Util.getType(stateUsage).getName();
		} else {
			return null;
		}
	}

	@Override
	public Object getFailureModeAttribute(Object errorState) {
		if (errorState instanceof final StateUsage stateUsage && ModelUtil.isErrorState(stateUsage)) {
			final Usage probabilityUsage = typeSelect(stateUsage.getMembers(), UsageImplCustom.class).stream()
				.filter(usage -> "property".equals(usage.getRedefinitionName()))
				.findFirst().orElse(null);
			if (probabilityUsage != null && probabilityUsage.getValue() != null) {
				final Expression expression = probabilityUsage.getValue().getExpression();
				if (expression instanceof final ReferenceExpression reference) {
					return reference.getReferencedElement();
				}
			}
		}
		return null;
	}

	@Override
	public EList<String> getFailureModeFields(Object errorState) {
		if (errorState instanceof final StateUsage stateUsage && ModelUtil.isErrorState(stateUsage)) {
			return typeSelect(stateUsage.getMembers(), UsageImplCustom.class).stream()
					.filter(usage -> !"probability".equals(usage.getRedefinitionName()) &&
							!"probability".equals(usage.getRedefinitionName()))
					.map(Usage::getName)
					.collect(Collectors.toCollection(ECollections::newBasicEList));
		}
		return emptyEList();
	}

	@Override
	public String getFailureModeValue(Object errorState, String field) {
		if (errorState instanceof final StateUsage stateUsage && ModelUtil.isErrorState(stateUsage)) {
			final Usage fieldUsage = typeSelect(stateUsage.getMembers(), UsageImplCustom.class).stream()
				.filter(usage -> field.equals(usage.getRedefinitionName()))
				.findFirst().orElse(null);
			if (fieldUsage != null && fieldUsage.getValue() != null) {
				return SysMLv2ToSMVExpression.getInstance(Language.LTL).serialize(fieldUsage.getValue().getExpression());
			}
		}
		return null;
	}

	@Override
	public EList<?> getOwnerDanglingInputPorts(Object stateMachine) {
		final var list = getOwnerNonStaticPortsFiltering(stateMachine, ModelUtil::isInputParameter, ModelUtil::isHavocGeneratedAttribute);
		list.addAll(getOwnerNonStaticPortsFiltering(stateMachine, ModelUtil::isInputParameter, param -> SysMLv2Util.containsMetadataUsagesOfType(param, "ActivityParam")));
		return list;
	}
}
