/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.api.core;

import static com.google.common.base.Predicates.and;
import static java.util.function.Predicate.not;
import static org.eclipse.emf.common.util.ECollections.asEList;
import static org.eclipse.emf.common.util.ECollections.emptyEList;
import static org.eclipse.emf.common.util.ECollections.newBasicEList;
import static org.eclipse.emf.common.util.ECollections.toEList;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.emf.common.util.ECollections;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.EcoreUtil2;

import com.google.common.collect.Iterables;
import com.google.common.graph.Traverser;

import eu.fbk.eclipse.explodtwin.api.exception.NotAPartException;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil.Bound;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil.ComponentsRelationship;
import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression;
import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression.Language;
import eu.fbk.eclipse.standardtools.utils.core.model.AbstractSystemModelClass;
import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.sysmlv2.sysMLv2.AssertionAction;
import eu.fbk.sysmlv2.sysMLv2.AttributeUsage;
import eu.fbk.sysmlv2.sysMLv2.Binding;
import eu.fbk.sysmlv2.sysMLv2.BoundedExpression;
import eu.fbk.sysmlv2.sysMLv2.Calculation;
import eu.fbk.sysmlv2.sysMLv2.ConnectionElement;
import eu.fbk.sysmlv2.sysMLv2.ConnectionUsage;
import eu.fbk.sysmlv2.sysMLv2.ConstraintUsage;
import eu.fbk.sysmlv2.sysMLv2.Container;
import eu.fbk.sysmlv2.sysMLv2.Definition;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Direction;
import eu.fbk.sysmlv2.sysMLv2.EnumDefinition;
import eu.fbk.sysmlv2.sysMLv2.Expression;
import eu.fbk.sysmlv2.sysMLv2.Feature;
import eu.fbk.sysmlv2.sysMLv2.ListExpression;
import eu.fbk.sysmlv2.sysMLv2.Model;
import eu.fbk.sysmlv2.sysMLv2.Multiplicity;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.Part;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.PortUsage;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.ReturnStatement;
import eu.fbk.sysmlv2.sysMLv2.Usage;
import eu.fbk.sysmlv2.sysMLv2.UsageChainExpression;
import eu.fbk.sysmlv2.sysMLv2.UsageExpression;
import eu.fbk.sysmlv2.sysMLv2.UsageReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.Value;
import eu.fbk.sysmlv2.util.SysMLv2Util;

public class SysMLv2SystemModel extends AbstractSystemModelClass {

	private final Map<String, Calculation> defineTransformationMap = new HashMap<>();
	private final Map<ReferenceExpression, Pair<Calculation, List<Expression>>> referenceMap = new HashMap<>();

	@Override
	public Object getNearestOwnerComponent(Object element) {
		if (element instanceof final EObject eObject) {
			return EcoreUtil2.getContainerOfType(eObject, PartDefinition.class);
		} else {
			return null;
		}
	}

	@Override
	public EList<String> getEnumValuesFromAttributes(Object component) {
		return ModelUtil.getAttributeUsagesInPart(component).stream()
				.map(usage -> typeSelect(SysMLv2Util.getAllTypes(usage), EnumDefinition.class).stream().findFirst().orElse(null))
				.filter(Objects::nonNull)
				.<String>mapMulti((enumDef, consumer) -> ModelUtil.getEnumValues(enumDef, true).forEach(consumer::accept))
				.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public String[] getValuesForEnumeratorType(Object object) {
		if (object instanceof final EnumDefinition enumDefinition) {
			return ModelUtil.getEnumValues(enumDefinition, true).toArray(String[]::new);
		}
		return null;
	}

	@Override
	public String getComponentTypeName(Object component) {
		return getName(getComponentType(component));
	}

	@Override
	public String getComponentInstanceName(Object component) {
		if (component instanceof final PartUsage partUsage) {
			return getName(partUsage);
		}

		throw new RuntimeException("Unexpected argument type");
	}

	@Override
	public Object getComponentType(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		return ModelUtil.getComponentType(part);
	}

	@Override
	public String getComponentInstanceTypeName(Object component) {
		return getComponentTypeName(component);
	}

	@Override
	public String[] getSubComponentsName(Object component) {
		return getSubComponentsInstances(component).stream()
			.map(this::getName)
			.distinct()
			.toArray(String[]::new);
	}

	@Override
	public Object getSubComponent(Object component, String subCompName) {
		if (component instanceof final Container container) {
			return SysMLv2Util.getMemberByName(container, PartUsage.class, subCompName);
		} else {
			return null;
		}
	}

	@Override
	public String getComponentName(Object component) {
		return getName(component);
	}

	@Override
	public EList<PartUsage> getSubComponentsInstances(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		final List<PartUsage> instances = typeSelect(partDefinition.getMembers(), PartUsage.class);
		instances.removeIf(ModelUtil::isActivity);
		return toEList(instances);
	}

	private List<String> getInvariantsForBoundedAttributes(Object component, String language) {
		final List<String> invariants = new ArrayList<>();
		final String and = Language.fromString(language) == Language.CLEANC ? " && " : " & ";
		ModelUtil.getAttributeUsagesInPart(component).stream()
			.filter(attribute -> ModelUtil.isIntervalType(SysMLv2Util.getType(attribute)))
			.forEach(attribute -> {
				final String name = attribute.getName();
				final String minValue = ModelUtil.getBoundValue(attribute, Bound.MIN);
				final String maxValue = ModelUtil.getBoundValue(attribute, Bound.MAX);
				final String lowerBound = name + " >= " + minValue;
				final String upperBound = name + " <= " + maxValue;
				invariants.add(lowerBound + and + upperBound);
			});
		return invariants;
	}

	private List<String> getExplicitInvariants(Object component, String language) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final Definition definition = ModelUtil.getComponentType(part);
		return typeSelect(definition.getMembers(), AttributeUsage.class).stream()
			.filter(ModelUtil::isInvariant)
			.map(attribute -> typeSelect(attribute.getMembers(), ConstraintUsage.class).get(0))
			.map(assertion -> assertion.getValue().getExpression())
			.map(expression -> SysMLv2ToSMVExpression.getInstance(language).serialize(expression))
			.toList();
	}

	@Override
	public String[] getComponentInvariants(Object component, String language) {
		final List<String> invariants = getInvariantsForBoundedAttributes(component, language);
		invariants.addAll(getExplicitInvariants(component, language));
		return invariants.toArray(String[]::new);
    }

	@Override
	public EList<DirectedElement> getNonStaticInputPorts(Object component) {
		return getNonStaticPorts(component).stream()
			.filter(ModelUtil::isInputParameter)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public EList<DirectedElement> getNonStaticOutputPorts(Object component) {
		return getNonStaticPorts(component).stream()
			.filter(ModelUtil::isOutputParameter)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public EList<DirectedElement> getNonStaticGenericPorts(Object component) {
		return getNonStaticPorts(component).stream()
			.filter(ModelUtil::isBidirectionalParameter)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public EList<DirectedElement> getNonStaticPorts(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final var ports = ModelUtil.getNonStaticPorts(part);
		ports.removeIf(port -> port.getName().endsWith("_done_input"));
		return ports;
	}

	//TODO: to be implemented after deciding how static ports are marked in SysMLv2
	@Override
	public EList<DirectedElement> getStaticPorts(Object component) {
		return emptyEList();
	}

	@Override
	public String getName(Object object) {
		String name = null;
		if (object instanceof Model) {
			name = "Model";
		} else if (object instanceof final NamedElement namedElement) {
			name = namedElement.getName();
			if (name != null) {
				name = name.replace("~", "Conjugate");
			}
		}
		return name;
	}

	@Override
	public String getPortName(Object port) {
		return getName(port);
	}

	@Override
	public Object getPortType(Object port) {
		if (port instanceof final Usage usage) {
			return ModelUtil.getUnboundedType(usage);
		} else {
			return null;
		}
	}

	@Override
	public boolean isBooleanType(Object type) {
		return ModelUtil.isBooleanType(type);
	}

	@Override
	public boolean isIntType(Object type) {
		return ModelUtil.isIntegerType(type);
	}

	@Override
	public boolean isIdealClockType(Object type) {
		return ModelUtil.isClockType(type);
	}

	@Override
	public boolean isRealType(Object type) {
		return ModelUtil.isRealType(type);
	}

	@Override
	public boolean isEnumType(Object type) {
		return ModelUtil.isEnumType(type);
	}

	@Override
	public boolean isEventType(Object type) {
		return type instanceof final Definition definition && SysMLv2Util.isEventDefinition(definition);
	}

	/*
	 * The default implementation of this method already returns false.
	 * This overriding is not a mistake, it is just here for clarity.
	 * This class is only used for translation into OSS, and for this
	 * purpose, we map both BoundedInt and BoundedReal (which are the
	 * only two bounded/interval/range types we support) into Integer and
	 * Real, respectively, plus invariants. Thus, we indeed want no type
	 * to be considered as a range type during translation.
	 */
	@Override
	public boolean isRangeType(Object object) {
		return false;
	}

	@Override
	public EList<ConnectionElement> getConnectionsPorts(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		return typeSelect(partDefinition.getMembers(), ConnectionElement.class).stream()
			/*
			 * Here we use connection.get{Source,Target}() instead of getConnector{Source,Target}(connection)
			 * as the latter assumes to receive a processed connection, i.e. a connection
			 * that does NOT connect two Port Usages (these connections require to be processed
			 * to connect the ports' internal members).
			 */
			.filter(connection -> Stream.of(connection.getSource(), connection.getTarget())
					.map(this::getConnectorEndOwner).filter(Part.class::isInstance).map(Part.class::cast)
					.noneMatch(ModelUtil::isActivity))
			.<ConnectionElement>mapMulti((connection, acceptor) -> {
				if (connection.getSource() instanceof final ReferenceExpression reference &&
						reference.getReferencedElement() instanceof PortUsage) {
					ModelUtil.generateConnectionsForFlattenedPort(connection).forEach(acceptor::accept);
				} else {
					acceptor.accept(connection);
				}
			})
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public String getConnectorEndName(Object connectorEnd) {
		if (connectorEnd instanceof final ReferenceExpression reference) {
			return getName(reference.getReferencedElement());
		} else if (connectorEnd instanceof final Expression expression) {
			return SysMLv2ToSMVExpression.getInstance(Language.LTL).serialize(expression);
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public Object getConnectorEndOwner(Object connectorEnd) {
		return ModelUtil.getConnectorEndOwner(connectorEnd);
	}

	private enum ConnectorEnd {
		SOURCE, TARGET;
	}

	private ReferenceExpression getBindingEnd(Binding binding, ConnectorEnd end) {
		final ReferenceExpression sourceReference = (ReferenceExpression) binding.getSource();
		final ReferenceExpression targetReference = binding.getTarget();
		final NamedElement source = sourceReference.getReferencedElement();
		final NamedElement target = targetReference.getReferencedElement();
		final Direction direction = ((DirectedElement) (source)).getDirection();
		final PartDefinition sourceOwner = EcoreUtil2.getContainerOfType(source, PartDefinition.class);
		final PartDefinition targetOwner = EcoreUtil2.getContainerOfType(target, PartDefinition.class);
		final ComponentsRelationship relationship = ModelUtil.compareComponents(sourceOwner, targetOwner);
		return switch (end) {
		case SOURCE -> switch (relationship) {
			case FIRST_CONTAINS_SECOND -> direction == Direction.IN ? sourceReference : targetReference;
			case SECOND_CONTAINS_FIRST -> direction == Direction.OUT ? sourceReference : targetReference;
			case SAME -> null;
		};
		case TARGET -> switch (relationship) {
			case FIRST_CONTAINS_SECOND -> direction == Direction.OUT ? sourceReference : targetReference;
			case SECOND_CONTAINS_FIRST -> direction == Direction.IN ? sourceReference : targetReference;
			case SAME -> null;
		};
		};
	}

	@Override
	public Object getConnectorSource(Object object) {
		if (object instanceof final ConnectionUsage connection) {
			return connection.getSource();
		} else if (object instanceof final Binding binding) {
			return getBindingEnd(binding, ConnectorEnd.SOURCE);
		} else {
			throw new IllegalArgumentException("Unexpected argument type");
		}
	}

	@Override
	public Object getConnectorTarget(Object object) {
		if (object instanceof final ConnectionUsage connection) {
			return connection.getTarget();
		} else if (object instanceof final Binding binding) {
			return getBindingEnd(binding, ConnectorEnd.TARGET);
		} else {
			throw new IllegalArgumentException("Unexpected argument type");
		}
	}

	@Override
	public EList<String> getAttributesNames(Object component) {
		return ModelUtil.getAttributeUsagesInPart(component).stream()
			.map(this::getName)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public Object getAttributeType(Object property) {
		return property instanceof final Usage usage ? ModelUtil.getUnboundedType(usage) : null;
	}

	@Override
	public EList<PartDefinition> getComponents(Object object) {
		if (object instanceof final EObject eObject) {
			return ModelUtil.getAllContentsOfTypeFromModel(EcoreUtil2.getResourceSet(eObject), PartUsage.class).stream()
					.map(ModelUtil::getComponentType)
					.filter(not(ModelUtil::isActivity))
					.distinct()
					.collect(Collectors.toCollection(ECollections::newBasicEList));
		} else {
			return null;
		}
	}

	@Override
	public EList<Feature> getFormulaConstraints(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		return typeSelect(partDefinition.getMembers(), Feature.class).stream()
			.filter(not(ModelUtil::isOfContractType))
			.filter(feature -> feature.getValue() != null && feature.getRedefinition() != null)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public String getFormulaConstraintText(Object constraint) {
		if (constraint instanceof final Feature feature) {
			final StringBuilder buffer = new StringBuilder();
			final String featureName = feature.getName();
			final Value featureValue = feature.getValue();
			final String value = SysMLv2ToSMVExpression.getInstance(Language.LTL).serialize(featureValue.getExpression());
			buffer.append(featureName).append(":=").append(value);
			return buffer.toString();
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public EList<Usage> getContractsOfComponent(Object component) {
		if (component instanceof final PartUsage partUsage) {
			return ModelUtil.getContractUsages(ModelUtil.getPartDefinition(partUsage));
		} else if (component instanceof final PartDefinition partDefinition) {
			return ModelUtil.getContractUsages(partDefinition);
		} else {
			throw new RuntimeException("Unexpected argument type");
		}
	}

	@Override
	public String getContractDefinitionsText(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		final EList<Usage> contracts = ModelUtil.getContractUsages(partDefinition);
		final StringBuilder buffer = new StringBuilder();
		contracts.forEach(contract -> buffer.append(convertContractToString(contract)));
		return buffer.toString();
	}

	private enum ConstraintType {
		ASSUMPTION, GUARANTEE;

		@Override
		public String toString() {
			return this.name().toLowerCase();
		}
	}

	private String convertContractToString(Usage contract) {
		final String name = contract.getName();
		final String assume = getFormalPropertyStrFromSysMLContract(contract, ConstraintType.ASSUMPTION);
		final String guarantee = getFormalPropertyStrFromSysMLContract(contract, ConstraintType.GUARANTEE);
		final String contractBody = "CONTRACT " + name + " assume : " + assume + " ; guarantee : " + guarantee + " ;";
		return contractBody;
	}

	private String getFormalPropertyStrFromSysMLContract(Usage contract, ConstraintType constraintType) {
		return typeSelect(contract.getMembers(), AssertionAction.class).stream()
			.map(assertion -> assertion.getInlineConstraint() != null ? assertion.getInlineConstraint() : assertion.getConstraint())
			.filter(constraintUsage -> constraintType.toString().equals(getName(constraintUsage)))
			.map(this::getConstraintAsText)
			.findFirst().orElse(null);
	}

	private String getConstraintAsText(ConstraintUsage constraint) {
		final Expression expression = constraint.getExpression();
		if (expression != null) {
			return SysMLv2ToSMVExpression.getInstance(Language.LTL).serialize(expression);
		} else {
			return "true";
		}
	}

	@Override
	public Object getContract(Object component, String contractName) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition definition = ModelUtil.getComponentType(part);
		return ModelUtil.getContractUsages(definition).stream()
				.filter(usage -> contractName.equals(getName(usage)))
				.findFirst().orElse(null);
	}

	@Override
	public EList<Expression> getContractRefinements(Object contract) {
		if (contract instanceof final Usage usage) {
			final Feature refinement = typeSelect(usage.getMembers(), Feature.class).stream()
				.filter(feature -> "refinedBy".equals(getName(feature)))
				.findFirst().orElse(null);
			if (refinement == null) {
				return emptyEList();
			}
			final Expression valueExpression = refinement.getValue().getExpression();
			if (valueExpression instanceof final ListExpression listExpression) {
				return listExpression.getExpressions();
			} else if (valueExpression instanceof final BoundedExpression boundedExpression) {
				return asEList(boundedExpression.getExpression());
			}
		}
		return null;
	}

	@Override
	public EList<Usage> getComponentInstancesOfContractRefinement(Object contractRefinement) {
		final EList<Usage> refinements = newBasicEList();
		if (contractRefinement instanceof final UsageChainExpression usageChainExpression) {
			final UsageExpression body = usageChainExpression.getBody();
			if (body instanceof final UsageReferenceExpression usageReferenceExpression) {
				refinements.add((Usage) usageReferenceExpression.getReferencedElement());
			} else if (body instanceof final UsageChainExpression usageChainExpression2) {
				refinements.addAll(extractAllUsages(usageChainExpression2));
			}
		}
		return refinements;
	}

	private List<Usage> extractAllUsages(UsageChainExpression usageChainExpression) {
		final List<Usage> list = new ArrayList<>();
		while (usageChainExpression.getBody() instanceof final UsageChainExpression bodyChain) {
			list.add((Usage) usageChainExpression.getReferencedElement());
			usageChainExpression = bodyChain;
		}
		list.add((Usage) usageChainExpression.getReferencedElement());
		list.add((Usage) usageChainExpression.getBody().getReferencedElement());
		return list;
	}

	@Override
	public Object getContractInstanceOfContractRefinement(Object contractRefinement) {
		if (contractRefinement instanceof final UsageChainExpression usageChainExpression) {
			return usageChainExpression.getReferencedElement();
		}
		return null;
	}

	@Override
	public String getContractInstanceName(Object contractInstance) {
		return getName(contractInstance);
	}

	@Override
	public boolean isAsyncComponent(Object component) {
		return getSubComponentsInstances(component).stream().anyMatch(ModelUtil::isAsyncPartUsage);
	}

	@Override
	public EList<Calculation> getUninterpretedFunctions(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		return EcoreUtil2.getAllContentsOfType(partDefinition, ReferenceExpression.class).stream()
			.map(ReferenceExpression::getReferencedElement)
			.filter(Calculation.class::isInstance).map(Calculation.class::cast)
			.filter(ModelUtil::isUninterpretedFunction)
			.distinct()
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public String getUninterpretedFunctionName(Object uninterpretedFunction) {
		if (uninterpretedFunction instanceof final Calculation calculation) {
			return getName(calculation);
		}
		throw new RuntimeException("Unexpected argument type");
	}

	public Feature getUninterpretedFunctionReturnedFeature(Object uninterpretedFunction) {
		if (uninterpretedFunction instanceof final Calculation calculation) {
			final EObject result = calculation.getResult();
			if (result instanceof final ReturnStatement returnStatement) {
				return returnStatement.getFeature();
			}
		}
		throw new RuntimeException("Unexpected argument type");
	}

	@Override
	public Definition getUninterpretedFunctionOutputType(Object uninterpretedFunction) {
		final Feature feature = getUninterpretedFunctionReturnedFeature(uninterpretedFunction);
		return SysMLv2Util.getType(feature);
	}

	public Stream<DirectedElement> getUninterpretedFunctionInputs(Object uninterpretedFunction) {
		if (uninterpretedFunction instanceof final Calculation calculation) {
			return typeSelect(calculation.getMembers(), DirectedElement.class).stream()
				.filter(and(ModelUtil::isInputParameter, directedElement -> directedElement.getTyping() != null));
		}
		throw new RuntimeException("Unexpected argument type");
	}

	@Override
	public EList<Definition> getUninterpretedFunctionInputTypes(Object uninterpretedFunction) {
		return getUninterpretedFunctionInputs(uninterpretedFunction)
			.map(input -> {
				final var type = ModelUtil.getUnboundedType(input);
				return type instanceof EnumDefinition ? ModelUtil.INTEGER_DEFINITION : type;
			})
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	private String[] multiplicityAsStringArray(Multiplicity multiplicity) {
		if (multiplicity == null) {
			return new String[] {"1", "1"};
		} else {
			final Integer lowerBound = multiplicity.getLowerBound();
			final Integer upperBound = multiplicity.getUpperBound();
			return new String[] {lowerBound.toString(), upperBound != null ? upperBound.toString() : lowerBound.toString()};
		}
	}

	@Override
	public String[] getUninterpretedFunctionOutputMultiplicity(Object uninterpretedFunction) {
		final Feature feature = getUninterpretedFunctionReturnedFeature(uninterpretedFunction);
		final Multiplicity multiplicity = feature.getMultiplicity();
		return multiplicityAsStringArray(multiplicity);
	}

	@Override
	public EList<String[]> getUninterpretedFunctionInputMultiplicities(Object uninterpretedFunction) {
		return getUninterpretedFunctionInputs(uninterpretedFunction)
			.map(DirectedElement::getMultiplicity)
			.map(this::multiplicityAsStringArray)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public String[] getPortMultiplicityBoundaries(Object port) {
		if (port instanceof final Usage usage) {
			return multiplicityAsStringArray(usage.getMultiplicity());
		} else {
			return null;
		}
	}

	@Override
	public boolean isInputPort(Object port) {
		return ModelUtil.isInputParameter(port);
	}

	@Override
	public boolean isOutputPort(Object port) {
		return ModelUtil.isOutputParameter(port);
	}

	@Override
	public boolean isInOutPort(Object port) {
		return ModelUtil.isBidirectionalParameter(port);
	}

	@Override
	public EList<?> getInterfaceAssertions(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition partDefinition = ModelUtil.getComponentType(part);
		return typeSelect(partDefinition.getMembers(), AttributeUsage.class).stream()
			.filter(this::isInterfaceAssertion)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	@Override
	public boolean isInterfaceAssertion(Object object) {
		return ModelUtil.isInterfaceAssertion(object);
	}

	@Override
	public String getInterfaceAssertionName(Object interfaceAssertion) {
		if (interfaceAssertion instanceof final AttributeUsage attributeUsage) {
			return getName(attributeUsage);
		} else {
			throw new RuntimeException("Invalid interface assertion, AttributeUsage expected");
		}
	}

	@Override
	public String getInterfaceAssertionBody(Object interfaceAssertion, String language) {
		if (interfaceAssertion instanceof final AttributeUsage attributeUsage) {
			final Usage formula = typeSelect(attributeUsage.getMembers(), Usage.class).stream()
				.filter(member -> "formula".equals(getName(member)))
				.findAny().orElseThrow(() -> new RuntimeException("No constraint found in interface assertion"));
			return SysMLv2ToSMVExpression.getInstance(language).serialize(formula.getValue().getExpression());
		} else {
			throw new RuntimeException("Invalid interface assertion, AttributeUsage expected");
		}
	}

	@Override
	public String getDefineName(Object define) {
		if (define instanceof Calculation || define instanceof ConstraintUsage) {
			return ((NamedElement) define).getName();
		} else {
			throw new IllegalArgumentException("Unexpected argument type");
		}
	}

	@Override
	public String getDefineBody(Object define, String language) {
		Expression expression = null;
		if (define instanceof final Calculation calculation) {
			final EObject result = calculation.getResult();
			if (result instanceof final Expression resultExpression) {
				expression = resultExpression;
			}
		} else if (define instanceof final ConstraintUsage constraintUsage) {
			expression = constraintUsage.getExpression();
		}
		if (expression != null) {
			return SysMLv2ToSMVExpression.getInstance(language).serialize(expression);
		} else {
			throw new IllegalArgumentException("Unexpected argument type");
		}
	}

	private static boolean isReferenceToInterpretedFunction(ReferenceExpression reference) {
		return reference.getReferencedElement() instanceof Calculation calculation &&
				ModelUtil.isInterpretedFunction(calculation);
	}

	private String computeProcessedCalculationName(Calculation calculation, List<Expression> arguments) {
		final SysMLv2ToSMVExpression serializer = SysMLv2ToSMVExpression.getInstance(Language.LTLSMV);
		final StringBuilder newNameBuilder = new StringBuilder(calculation.getName());
		arguments.forEach(arg -> {
			newNameBuilder.append("_");
			newNameBuilder.append(serializer.serialize(arg).replaceAll("[^a-zA-Z0-9]", ""));
		});
		return newNameBuilder.toString();
	}

	/*
	 * This method accepts an InvocationTreeNode instance whose effectiveArguments list
	 * has already been populated. It then generates a clone of the invoked Calculation
	 * where each parameter has been replaced with its respective effective argument.
	 * The generated Calculation is stored in the passed InvocationTreeNode and it is
	 * also returned.
	 */
	@SuppressWarnings("unchecked")
	private Calculation invocationToCalculation(InvocationTreeNode node) {
		final String cloneName = computeProcessedCalculationName(node.invokedCalculation, node.effectiveArguments);
		/*
		 * TODO This code is for using the defineTransformationMap as a cache
		 * and avoid unneeded regeneration of duplicate processed Calculations.
		 * However, it is necessary to devise a way to invalidate the cache entries
		 * if their expression is modified in the editor.
		 */
//		final Calculation cachedCalculation = defineTransformationMap.get(cloneName);
//		if (cachedCalculation != null) {
//			node.processedCalculation = cachedCalculation;
//			return cachedCalculation;
//		}
		final Calculation clone = EcoreUtil.copy(node.invokedCalculation);
		clone.setName(cloneName);
		EcoreUtil2.getAllContentsOfType(clone.getResult(), ReferenceExpression.class).stream()
			.forEach(reference -> {
				final int index = typeSelect(clone.getMembers(), NamedElement.class).indexOf(reference.getReferencedElement());
				if (index != -1) {
					final Expression effectiveArgument = node.effectiveArguments.get(index);
					if (effectiveArgument instanceof final ReferenceExpression effectiveArgumentAsReference) {
						reference.setReferencedElement(effectiveArgumentAsReference.getReferencedElement());
					} else {
						final var container = reference.eContainer().eGet(reference.eContainingFeature());
						if (container instanceof final List<?> arguments) {
							final var argumentsList = (List<Expression>) arguments;
							final int effectiveArgumentIndex = argumentsList.indexOf(reference);
							argumentsList.set(effectiveArgumentIndex, effectiveArgument);
						} else {
							reference.eContainer().eSet(reference.eContainingFeature(), effectiveArgument);
						}
					}
				}
			});
		node.processedCalculation = clone;
		defineTransformationMap.put(clone.getName(), clone);
		return clone;
	}

	private void populateEffectiveArguments(InvocationTreeNode node) {
		final List<Expression> argumentsFromParseTree = node.reference.getArguments();
		node.effectiveArguments = new ArrayList<>(argumentsFromParseTree.size());
		final InvocationTreeNode parent = node.parent;
		if (parent == null) { // This is the easy case, i.e. the tree root
			argumentsFromParseTree.forEach(argument -> node.effectiveArguments.add(EcoreUtil.copy(argument)));
		} else {
			final List<NamedElement> parentParameters = typeSelect(parent.invokedCalculation.getMembers(), NamedElement.class);
			for (int i = 0; i < argumentsFromParseTree.size(); i++) {
				final Expression argument = argumentsFromParseTree.get(i);
				if (argument instanceof final ReferenceExpression argumentAsReference) {
					final int indexInParentParameters = parentParameters.indexOf(argumentAsReference.getReferencedElement());
					node.effectiveArguments.add(EcoreUtil.copy(parent.effectiveArguments.get(indexInParentParameters)));
				} else {
					node.effectiveArguments.add(EcoreUtil.copy(argument));
				}
			}
		}
	}

	@Override
	public EList<EObject> getDefines(Object component) {
		if (!(component instanceof final Part part)) {
			throw new NotAPartException(component.getClass());
		}
		final PartDefinition definition = ModelUtil.getComponentType(part);
		final List<ReferenceExpression> interpretedFunctionsCallsWithoutArguments = EcoreUtil2.getAllContentsOfType(definition, ReferenceExpression.class).stream()
				.filter(SysMLv2SystemModel::isReferenceToInterpretedFunction)
				.filter(call -> call.getArguments().isEmpty())
				.toList();
		final List<ReferenceExpression> interpretedFunctionsCallsWithArguments = EcoreUtil2.getAllContentsOfType(definition, ReferenceExpression.class).stream()
				.filter(SysMLv2SystemModel::isReferenceToInterpretedFunction)
				.filter(call -> !call.getArguments().isEmpty())
				.toList();

		/*
		 * This is the list the method will return. We start by initialising it with all
		 * the nonparametrised interpreted functions used by the current component.
		 */
		final EList<Calculation> defines = interpretedFunctionsCallsWithoutArguments.stream()
				.map(ReferenceExpression::getReferencedElement)
				.filter(Calculation.class::isInstance).map(Calculation.class::cast)
				.collect(Collectors.toCollection(ECollections::newBasicEList));

		/*
		 * Now all the invocation trees, and a traverser, are created.
		 */
		final List<InvocationTreeNode> invocationTreesRoots = interpretedFunctionsCallsWithArguments.stream()
			.map(InvocationTreeNode::new)
			.toList();
		final Traverser<InvocationTreeNode> traverser = Traverser.<InvocationTreeNode>forTree(
				InvocationTreeNode::getNestedInvocations);

		/*
		 * For each node in each tree, populate its effectiveArguments list.
		 */
		invocationTreesRoots.forEach(rootNode -> traverser.breadthFirst(rootNode).forEach(this::populateEffectiveArguments));

		/*
		 * Again, for each node in each tree, generate the Calculation corresponding to the
		 * invocation and process all other function invocations contained in its expression.
		 */
		invocationTreesRoots.forEach(rootNode -> {
			traverser.depthFirstPostOrder(rootNode).forEach(treeNode -> {
				final Calculation clone = EcoreUtil.copy(invocationToCalculation(treeNode));
				EcoreUtil2.getAllContentsOfType(clone.getResult(), ReferenceExpression.class).stream()
					.filter(ReferenceExpression::isInvocation)
					.filter(SysMLv2SystemModel::isReferenceToInterpretedFunction)
					.forEach(reference -> {
						/*
						 * Retrieve the processed Calculation from the map, where
						 * invocationToCalculation(InvocationTreeNode) has inserted
						 * it previously.
						 */
						final String mapKey = computeProcessedCalculationName(
								(Calculation) reference.getReferencedElement(), reference.getArguments());
						final Calculation targetCalculation = defineTransformationMap.get(mapKey);
						setReferenceToProcessedCalculation(reference, targetCalculation);
					});
				if (defines.stream().noneMatch(define -> define.getName().equals(clone.getName()))) {
					defines.add(clone);
				}
			});
			referenceMap.put(rootNode.reference, new Pair<Calculation, List<Expression>>(
					(Calculation) rootNode.reference.getReferencedElement(), List.copyOf(rootNode.reference.getArguments())));
			setReferenceToProcessedCalculation(rootNode.reference, rootNode.processedCalculation);
		});

		final EList<ConstraintUsage> constraints = toEList(typeSelect(definition.getMembers(), ConstraintUsage.class));
		constraints.removeIf(constraint -> (!"precondition".equals(constraint.getName()) &&
			!"postcondition".equals(constraint.getName())) || constraint.isAbstract());
		final EList<EObject> definesList = toEList(Iterables.concat(defines, constraints));
		ModelUtil.DEFINES_MAP.put(definition.getName(), definesList);
		return definesList;
	}

	private void setReferenceToProcessedCalculation(ReferenceExpression reference, Calculation calculation) {
		reference.setReferencedElement(calculation);
		reference.getArguments().clear();
		reference.setInvocation(false);
	}

	/*
	 * Data structure representing an invocation of an interpreted function.
	 * It is designed to be used as a node in a tree of function invocations, e.g.
	 * an invocation of the function foo(), which in its return expression contains an invocation
	 * to the function bar() which, in turn, contains an invocation to baz() will eventually
	 * generate the following InvocationTreeNode tree.
	 *
	 * 			foo()
	 * 			/
	 * 		  bar()
	 * 		  /
	 * 		baz()
	 *
	 * The InvocationTreeNode constructor takes care of searching for nested invocations
	 * and instantiating their InvocationTreeNode objects.
	 *
	 * This class contains the following members:
	 * - reference: the reference to the ReferenceExpression grammar object related to this
	 * 				invocation. This grammar object is created when an invocation to an interpreted
	 * 				function is parsed in the textual model.
	 * - invokedCalculation: the Calculation grammar object that the invocation invokes.
	 * 						 This grammar object is created when an interpreted function definition
	 * 						 (e.g. a 'calc' or 'calc def') is parsed in the textual model.
	 * - nestedInvocations: the list of this invocation's child nodes in the tree.
	 * - effectiveArguments: the effective arguments passed to the function when it is invoked. Given
	 * 						 the following model excerpt:
	 *
	 * 						 calc def foo {
	 * 							 in x : Integer;
	 * 							 x + bar(x, 3)
	 * 						 }
	 * 						 calc def bar {
	 * 							 in m : Integer;
	 * 							 in n : Integer;
	 * 							 m + n
	 * 						 }
	 * 						 attribute baz : Integer := foo(10);
	 *
	 *						 the effective argument passed to foo is 10, while the effective arguments
	 *						 passed to bar are 10 and 3.
	 * - parent: this invocation's parent in the invocation tree.
	 * - processedCalculation: the generated Calculation object that represents the invocation.
	 */
	private final static class InvocationTreeNode {
		private final ReferenceExpression reference;
		private final Calculation invokedCalculation;
		private final List<InvocationTreeNode> nestedInvocations;
		private List<Expression> effectiveArguments;
		private InvocationTreeNode parent = null;
		private Calculation processedCalculation;

		public InvocationTreeNode(ReferenceExpression reference) {
			this.reference = reference;
			this.invokedCalculation = (Calculation) this.reference.getReferencedElement();
			this.nestedInvocations = EcoreUtil2.getAllContentsOfType(invokedCalculation.getResult(), ReferenceExpression.class).stream()
					.filter(SysMLv2SystemModel::isReferenceToInterpretedFunction)
					.map(InvocationTreeNode::new)
					.toList();
			this.nestedInvocations.forEach(invocation -> invocation.parent = this);
		}

		private List<InvocationTreeNode> getNestedInvocations() {
			return nestedInvocations;
		}
	}

}
