/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.api.ui;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import java.util.function.BiPredicate;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.validation.CheckMode;
import org.eclipse.xtext.validation.IResourceValidator;
import org.eclipse.xtext.validation.Issue;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Provider;

import eu.fbk.eclipse.explodtwin.api.core.ModelLibrary;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.sysmlv2.SysMLv2StandaloneSetup;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;

/**
 * This class is just an entry point to create a Java Application and then export it as a runnable JAR.
 * The JAR can be used by Python to access the top level node of the loaded model and use the ModelLibrary
 * to query it.
 *
 * Export instructions for Eclipse:
 * - create a Run Configuration for this class (rightclick Run As/Java Application
 * - File/Export... and select Runnable JAR File
 * - select the launch configuration just created
 * - Extract required libraries... (it seems that the Python library cannot read the libraries in JAR format)
 * - distribute the JAR (it can be tested with "java -jar xxx.jar)
 */
public class Wrapper {

	@Inject
	private Provider<ResourceSet> resourceSetProvider;

	@Inject
	private IResourceValidator validator;

	private static Wrapper instance = null;

	@Inject
	private Wrapper() {}

	public static Wrapper getInstance() {
		if (instance == null) {
			final Injector injector = new SysMLv2StandaloneSetup().createInjectorAndDoEMFRegistration();
			instance = injector.getInstance(Wrapper.class);
		}
		return instance;
	}

	public static void main(String[] args) throws Exception {
		final Wrapper wrapper = getInstance();
		String inputFilePath = "resources/in.sysml";
		if (args.length != 0) {
			inputFilePath = args[0];
		} else {
			System.err.println("No path to EMF resource provided, using default: " + inputFilePath);
		}
		final PartUsage systemComponent = wrapper.loadModel(inputFilePath);
		wrapper.navigateModel(systemComponent);
	}

	// Sample method to show how to navigate a model
	private void navigateModel(PartUsage systemComponent) {
		final ModelLibrary modelLibrary = new ModelLibrary();
		System.out.println("root = " + systemComponent);

		System.out.println("Subcomponents names = " + modelLibrary.getSubComponentsNames(systemComponent));

		List<PartUsage> comps = modelLibrary.getSubComponents(systemComponent);
		for (Object comp : comps) {
			System.out.println("subComp = " + comp);
		}

		StateUsage mainST = modelLibrary.getFirstNominalStateMachine(systemComponent);
		System.out.println("mainST = " + mainST);

		StateUsage st = modelLibrary.getFirstNominalStateMachine(comps.get(0));
		System.out.println("ST = " + st);
	}

	/**
	 * Returns the top-level component of the model.
	 * @param fileName
	 * @return
	 */
	public PartUsage loadModel(String fileName) {
		Resource resource = getResource(fileName);

		final List<Issue> issues = new ArrayList<>();
		EcoreUtil2.getResourceSet(resource).getResources().forEach(res ->
				issues.addAll(validator.validate(res, CheckMode.ALL, CancelIndicator.NullImpl)));
		issues.removeIf(issue -> !issue.getSeverity().equals(Severity.ERROR));
		if (!issues.isEmpty()) {
			final StringBuilder builder = new StringBuilder("Invalid model. Errors:\n");
			issues.forEach(error -> builder.append(error.toString() + "\n"));
			throw new RuntimeException(builder.toString());
		}

		EObject top = resource.getContents().get(0);
		return ModelUtil.getSystemPart(EcoreUtil2.getResourceSet(top));
	}

	public static List<Path> findSysMLFiles(String directory) {
		final BiPredicate<Path, BasicFileAttributes> matcher = (path, attributes) -> attributes.isRegularFile() && path.getFileName().toString().endsWith(".sysml");
		try (final var pathsStream = Files.find(Paths.get(directory), Integer.MAX_VALUE, matcher)) {
			return pathsStream.toList();
		} catch (IOException e) {
			return List.of();
		}
	}

	private Resource getResource(String inputFilePath) {
		ResourceSet set = resourceSetProvider.get();
		final int index = inputFilePath.lastIndexOf("/");
		final String userModelFolder = index == -1 ? "." : inputFilePath.substring(0, index);
		findSysMLFiles(userModelFolder)
			.forEach(path -> set.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		return set.getResources().stream()
			.filter(resource -> resource.getURI().toFileString().endsWith(inputFilePath))
			.findAny().orElseThrow();
	}

}
