/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.api.util;


import static com.google.common.base.Predicates.or;
import static java.util.function.Predicate.not;
import static org.eclipse.emf.common.util.ECollections.emptyEList;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.emf.common.util.ECollections;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.EcoreUtil2;

import com.google.common.collect.Streams;

import eu.fbk.eclipse.explodtwin.api.util.SysMLv2ToSMVExpression.Language;
import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.sysmlv2.sysMLv2.AttributeDefinition;
import eu.fbk.sysmlv2.sysMLv2.AttributeUsage;
import eu.fbk.sysmlv2.sysMLv2.Binding;
import eu.fbk.sysmlv2.sysMLv2.Calculation;
import eu.fbk.sysmlv2.sysMLv2.CalculationDefinition;
import eu.fbk.sysmlv2.sysMLv2.ConnectionElement;
import eu.fbk.sysmlv2.sysMLv2.Container;
import eu.fbk.sysmlv2.sysMLv2.Definition;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Direction;
import eu.fbk.sysmlv2.sysMLv2.EnumDefinition;
import eu.fbk.sysmlv2.sysMLv2.EnumUsage;
import eu.fbk.sysmlv2.sysMLv2.EqualityExpression;
import eu.fbk.sysmlv2.sysMLv2.Expression;
import eu.fbk.sysmlv2.sysMLv2.Feature;
import eu.fbk.sysmlv2.sysMLv2.ItemUsage;
import eu.fbk.sysmlv2.sysMLv2.Model;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.Operator;
import eu.fbk.sysmlv2.sysMLv2.Package;
import eu.fbk.sysmlv2.sysMLv2.Part;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.PortDefinition;
import eu.fbk.sysmlv2.sysMLv2.PortUsage;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.SysMLv2Factory;
import eu.fbk.sysmlv2.sysMLv2.Transition;
import eu.fbk.sysmlv2.sysMLv2.TypedElement;
import eu.fbk.sysmlv2.sysMLv2.Usage;
import eu.fbk.sysmlv2.sysMLv2.UsageChainExpression;
import eu.fbk.sysmlv2.util.SysMLv2Util;

public class ModelUtil {
	public static final String BOUNDED_REAL = "BoundedReal";
	public static final String BOUNDED_INT = "BoundedInt";
	public static final Map<String, EList<EObject>> DEFINES_MAP = new HashMap<>();
	public static final Map<Pair<DirectedElement, PortUsage>, DirectedElement> NESTED_TO_FLATTENED = new HashMap<>();
	private static final AttributeDefinition BOOLEAN_DEFINITION = SysMLv2Factory.eINSTANCE.createAttributeDefinition();
	public static final AttributeDefinition INTEGER_DEFINITION = SysMLv2Factory.eINSTANCE.createAttributeDefinition();
	private static final AttributeDefinition REAL_DEFINITION = SysMLv2Factory.eINSTANCE.createAttributeDefinition();
	public final static CalculationDefinition NEXT = SysMLv2Factory.eINSTANCE.createCalculationDefinition();

	static {
		BOOLEAN_DEFINITION.setName("Boolean");
		INTEGER_DEFINITION.setName("Integer");
		REAL_DEFINITION.setName("Real");
		NEXT.setName("next");
	}

	// Suppress default constructor for noninstantiability
	private ModelUtil() {
		throw new AssertionError();
	}

	public static boolean isInvariant(DirectedElement element) {
		return element instanceof final AttributeUsage attribute && SysMLv2Util.getType(attribute).getName().equals("Invar");
	}

	public static PartDefinition getPartDefinition(PartUsage partUsage) {
		try {
			return typeSelect(SysMLv2Util.getAllTypes(partUsage), PartDefinition.class).get(0);
		} catch (IndexOutOfBoundsException e) {
			return null;
		}
	}

	public static boolean isBooleanType(Object type) {
		return type instanceof final AttributeDefinition datatype && datatype.getName().equals("Boolean");
	}

	public static boolean isIntegerType(Object type) {
		return type instanceof final AttributeDefinition datatype &&
				(datatype.getName().equals("Integer") || datatype.getName().equals("Natural"));
	}

	public static boolean isRealType(Object type) {
		return type instanceof final AttributeDefinition datatype && datatype.getName().equals("Real");
	}

	public static boolean isEnumType(Object type) {
		return type instanceof EnumDefinition;
	}

	public static boolean isClockType(Object type) {
		return type instanceof final AttributeDefinition datatype && datatype.getName().equals("Clock");
	}

	public static EList<Usage> getAttributeUsagesInPart(Object object) {
		if (object instanceof final PartDefinition partDefinition) {
			final var attributes = getAttributeUsagesInContainer(partDefinition);
			if (!isActivity(partDefinition) && partDefinition.getSpecialization() != null) {
				partDefinition.getSpecialization().getReferences().forEach(reference ->
					attributes.addAll(getAttributeUsagesInPart(reference.getReferencedElement())));
			}
			return attributes;
		} else if (object instanceof final PartUsage partUsage) {
			return getAttributeUsagesInContainer(getPartDefinition(partUsage));
		} else {
			return emptyEList();
		}
	}

	private static EList<ItemUsage> getEventsInPart(Object object) {
		if (object instanceof final PartDefinition partDefinition) {
			final var events = getEventsInContainer(partDefinition);
			if (!isActivity(partDefinition) && partDefinition.getSpecialization() != null) {
				partDefinition.getSpecialization().getReferences().forEach(reference ->
					events.addAll(getEventsInPart(reference.getReferencedElement())));
			}
			return events;
		} else if (object instanceof final PartUsage partUsage) {
			return getEventsInContainer(getPartDefinition(partUsage));
		} else {
			return emptyEList();
		}
	}

	private static EList<Usage> getAttributeUsagesInContainer(Container container) {
		return container.getMembers().stream()
			.filter(or(AttributeUsage.class::isInstance, Feature.class::isInstance))
			.map(Usage.class::cast)
			.filter(usage -> usage.getValue() == null || usage.getRedefinition() == null)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	private static EList<ItemUsage> getEventsInContainer(Container container) {
		return container.getMembers().stream()
			.filter(ItemUsage.class::isInstance).map(ItemUsage.class::cast)
			.filter(or(SysMLv2Util::isEvent, ModelUtil::isBooleanParameter))
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	public static List<PortUsage> getPortUsagesInPart(Part part) {
		return typeSelect(getComponentType(part).getMembers(), PortUsage.class);
	}

	public static EList<String> getEnumValues(EnumDefinition enumDefinition, boolean mapToIntegers) {
		return typeSelect(enumDefinition.getMembers(), EnumUsage.class).stream()
			.map(enumUsage -> {
				if (enumUsage.getValue() != null && mapToIntegers) {
					return SysMLv2ToSMVExpression.getInstance(Language.CLEANC).serialize(enumUsage.getValue().getExpression());
				} else {
					return enumUsage.getName();
				}
			})
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	public static <T extends EObject> List<T> getAllContentsOfTypeFromModel(ResourceSet resourceSet, Class<T> type) {
		/*
		 *  We need a reference EObject that surely belongs to the model (and not to the libraries).
		 *  The model's system component is a good fit.
		 */
		final PartUsage systemComponent = getSystemPart(resourceSet);
		return Streams.stream(EcoreUtil.getAllContents(resourceSet, true))
				.filter(type::isInstance).map(type::cast)
				.filter(eObject -> eObjectsBelongToSameProject(eObject, systemComponent) &&
						!EcoreUtil2.getContainerOfType(eObject, Package.class).isLibrary())
				.collect(Collectors.toList());
	}

	private static boolean eObjectsBelongToSameProject(EObject eObject1, EObject eObject2) {
		final var uri1 = eObject1.eResource().getURI();
		final var uri2 = eObject2.eResource().getURI();
		if (uri1.isPlatform() && uri2.isPlatform()) {
			final String separator = "/";
			final String projectName1 = uri1.toPlatformString(false).split(separator)[1];
			final String projectName2 = uri2.toPlatformString(false).split(separator)[1];
			return projectName2.equals(projectName1);
		} else if (uri1.isFile() && uri2.isFile()) {
			return true;
		}
		return false;
	}

	public static EList<Usage> getContractUsages(PartDefinition partDefinition) {
		return getAttributeUsagesInContainer(partDefinition).stream()
			.filter(ModelUtil::isOfContractType)
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	enum SpecialType {
		CLOCK, CONTRACT
	}

	private static boolean isOfSpecialType(Usage usage, SpecialType type) {
		if (usage instanceof Feature || usage instanceof AttributeUsage) {
			if (usage.getRedefinition() != null) {
				usage = (Usage) usage.getRedefinition().getReferences().get(0).getReferencedElement();
			} else if (usage.getTyping() == null) {
				throw new IllegalArgumentException("Usage " + usage.getName() + " does not redefine anything and is not typed.");
			}
			return type == SpecialType.CLOCK ? isClockType(SysMLv2Util.getAllTypes(usage)) : isContractType(SysMLv2Util.getAllTypes(usage));
		} else {
			return false;
		}
	}

	public static boolean isOfContractType(Usage usage) {
		return isOfSpecialType(usage, SpecialType.CONTRACT);
	}

	public static boolean isOfClockType(Usage usage) {
		return isOfSpecialType(usage, SpecialType.CLOCK);
	}

	public static Set<StateUsage> getNominalStateMachinesOfComponent(Part part) {
		final PartDefinition partDefinition = getComponentType(part);
		return typeSelect(partDefinition.getMembers(), StateUsage.class).stream()
				.filter(ModelUtil::isNominalStateMachine)
				.collect(Collectors.toSet());
	}

	public static Model getModel(EObject element) {
		return (Model) EcoreUtil2.getRootContainer(element);
	}

	public static boolean isContractType(List<Definition> definitions) {
		return isOfTypeName(definitions, "Contract");
	}

	private static boolean isOfTypeName(List<Definition> types, String typeName) {
		return types.stream().anyMatch(type -> typeName.equals(type.getName()));
	}

	public static boolean isAsyncPartUsage(PartUsage partUsage) {
		return isOfTypeName(SysMLv2Util.getAllTypes(partUsage), "Asynchronous");
	}

	public static boolean isNominalStateMachine(StateUsage stateMachine) {
		return typeSelect(stateMachine.getMembers(), StateUsage.class).stream()
			.noneMatch(ModelUtil::isErrorState);
	}

	public static boolean isErrorState(StateUsage state) {
		return SysMLv2Util.getAllTypes(state).stream().anyMatch(ModelUtil::isErrorStateType);
	}

	public static boolean isErrorStateType(Definition type) {
		return "ErrorState".equals(type.getName()) || type.getSpecialization().getReferences().stream()
				.map(ReferenceExpression::getReferencedElement)
				.filter(Definition.class::isInstance).map(Definition.class::cast)
				.anyMatch(ModelUtil::isErrorStateType);
	}

	private static boolean portHasDirection(Object port, Direction direction) {
		return port instanceof final DirectedElement usage && direction.equals(usage.getDirection());
	}

	public static boolean isInputParameter(Object port) {
		return portHasDirection(port, Direction.IN);
	}

	public static boolean isOutputParameter(Object port) {
		return portHasDirection(port, Direction.OUT);
	}

	public static boolean isBidirectionalParameter(Object port) {
		return portHasDirection(port, Direction.INOUT);
	}

	public static boolean isBooleanParameter(Usage port) {
		return isBooleanType(SysMLv2Util.getType(port));
	}

	public static PartUsage getSystemPart(ResourceSet resourceSet) {
		return Streams.stream(EcoreUtil2.getAllContents(resourceSet, true))
			.filter(PartUsage.class::isInstance).map(PartUsage.class::cast)
			.filter(SysMLv2Util::isSystemPartUsage)
			.findAny().orElseThrow(() -> new NoSuchElementException("No SystemComponent found."));
	}

	public static boolean usesInfiniteDomainVariables(EObject eObject) {
		return getAllContentsOfTypeFromModel(EcoreUtil2.getResourceSet(eObject), Usage.class).stream()
			.filter(or(AttributeUsage.class::isInstance, Feature.class::isInstance))
			.anyMatch(ModelUtil::isInfiniteDomainVariable);
	}

	public static boolean isInfiniteDomainVariable(Usage usage) {
		return isIntegerType(getUnboundedType(usage)) || isRealType(getUnboundedType(usage));
	}

	public static boolean isUninterpretedFunction(Calculation calculation) {
		return SysMLv2Util.containsMetadataUsagesOfType(calculation, "UninterpretedFunction");
	}

	public static boolean isInterpretedFunction(Calculation calculation) {
		return !isUninterpretedFunction(calculation) &&
				/*
				 * The dummy 'next' calculation used in model post-processing and translation
				 * is not contained in any package, thus the following check.
				 */
				EcoreUtil2.getContainerOfType(calculation, Package.class) != null &&
				!EcoreUtil2.getContainerOfType(calculation, Package.class).isLibrary();
	}

	public static boolean isStateVariable(AttributeUsage attributeUsage) {
		return !isCalibrationParameter(attributeUsage) && isOutputParameter(attributeUsage);
	}

	public static boolean isCalibrationParameter(AttributeUsage attributeUsage) {
		return SysMLv2Util.containsMetadataUsagesOfType(attributeUsage, "CalibrationParam");
	}

	public static Definition getUnboundedType(TypedElement element) {
		Definition type = SysMLv2Util.getType(element);
		if (type == null) {
			return type;
		}
		if (isBoundedIntType(type)) {
			return INTEGER_DEFINITION;
		} else if (isBoundedRealType(type)) {
			return REAL_DEFINITION;
		} else {
			return type;
		}
	}

	public static boolean isInterfaceAssertion(Object object) {
		if (object instanceof final AttributeUsage attribute) {
			return isOfTypeName(SysMLv2Util.getAllTypes(attribute), "LTLSpec");
		} else {
			return false;
		}
	}

	public static boolean isActivity(Part part) {
		final PartDefinition partDefinition = getComponentType(part);
		return SysMLv2Util.partDefinitionIsActivity(partDefinition);
	}

	public static PartDefinition getComponentType(Part part) {
		if (part instanceof final PartUsage partUsage) {
			return getPartDefinition(partUsage);
		} else {
			return (PartDefinition) part;
		}
	}

	public static EList<EObject> getDefines(Part part) {
		final PartDefinition definition = getComponentType(part);
		return DEFINES_MAP.get(definition.getName());
	}

	public static DirectedElement flattenNestedMember(DirectedElement member, PortUsage portUsage) {
		final var key = new Pair<>(member, portUsage);
		DirectedElement flattenedMember = NESTED_TO_FLATTENED.get(key);
		if (flattenedMember == null) {
			flattenedMember = EcoreUtil.copy(member);
			final String flattenedName = flattenedMember.getName() + "__in__" + portUsage.getName();
			flattenedMember.setName(flattenedName);
			if (isOutputParameter(flattenedMember) && SysMLv2Util.isEvent(flattenedMember)) {
				flattenedMember.getTyping().getReferences().get(0).setReferencedElement(BOOLEAN_DEFINITION);
			}
			NESTED_TO_FLATTENED.put(key, flattenedMember);
		}
		return flattenedMember;
	}

	public static EList<DirectedElement> getNonStaticPorts(Part part) {
		getPortUsagesInPart(part).forEach(portUsage ->
				typeSelect(SysMLv2Util.getType(portUsage).getMembers(), DirectedElement.class).stream()
						.map(element -> flattenNestedMember(element, portUsage))
						.filter(flattenedMember -> flattenedMember.eContainer() == null)
						.forEach(flattenedMember -> EcoreUtil2.getContainerOfType(portUsage, PartDefinition.class).getMembers().add(flattenedMember)));
		final EList<Usage> attributeUsages = getAttributeUsagesInPart(part);
		final EList<DirectedElement> directedAttributes = typeSelect(attributeUsages, DirectedElement.class).stream()
			.filter(not(ModelUtil::isOfContractType))
			.filter(usage -> !Direction.NONE.equals(usage.getDirection()))
			.collect(Collectors.toCollection(ECollections::newBasicEList));
		directedAttributes.addAll(getEventsInPart(part));
		return directedAttributes;
	}

	public static boolean isSubsystem(Part part) {
		final PartDefinition partDefinition = getComponentType(part);
		return SysMLv2Util.partDefinitionIsSubsystem(partDefinition);
	}

	private static enum StateMachineType {
		MISSION("mission"), PLATFORM("platform");

		public final String name;

		StateMachineType(String name) {
			this.name = name + "StateMachines";
		}
	}

	private static StateUsage getStateMachineOfType(Part part, StateMachineType type) {
		final Definition definition = getComponentType(part);
		return typeSelect(definition.getMembers(), StateUsage.class).stream()
			.filter(state -> state.getSubsetting() != null &&
					state.getSubsetting().getReferences().get(0).getReferencedElement().getName().equals(type.name))
			.findAny().orElse(null);
	}

	public static StateUsage getPlatformStateMachine(Part part) {
		return getStateMachineOfType(part, StateMachineType.PLATFORM);
	}

	public static StateUsage getMissionStateMachine(Part part) {
		return getStateMachineOfType(part, StateMachineType.MISSION);
	}

	public static EList<ConnectionElement> getConnectionsPorts(Part part) {
		final PartDefinition partDefinition = getComponentType(part);
		return typeSelect(partDefinition.getMembers(), ConnectionElement.class).stream()
			.<ConnectionElement>mapMulti((connection, acceptor) -> {
				if (connection.getSource() instanceof final ReferenceExpression reference &&
						reference.getReferencedElement() instanceof PortUsage) {
					generateConnectionsForFlattenedPort(connection).forEach(acceptor::accept);
				} else {
					acceptor.accept(connection);
				}
			})
			.collect(Collectors.toCollection(ECollections::newBasicEList));
	}

	public static List<ConnectionElement> generateConnectionsForFlattenedPort(ConnectionElement connection) {
		final PortUsage source = (PortUsage) ((ReferenceExpression) connection.getSource()).getReferencedElement();
		final PortUsage target = (PortUsage) connection.getTarget().getReferencedElement();
		final var memberPairs = buildPortMemberPairs(source, target);
		final List<ConnectionElement> generatedConnections = new ArrayList<>();
		memberPairs.forEach(pair -> {
			final DirectedElement flattenedSource = flattenNestedMember(pair.getLeft(), source);
			final DirectedElement flattenedTarget = flattenNestedMember(pair.getRight(), target);
			final ConnectionElement clone = EcoreUtil.copy(connection);
			final ReferenceExpression sourceReference = (ReferenceExpression) clone.getSource();
			final ReferenceExpression targetReference = clone.getTarget();
			sourceReference.setReferencedElement(flattenedSource);
			targetReference.setReferencedElement(flattenedTarget);
			if (connection instanceof Binding) {
				if ((isInputParameter(flattenedSource) && isInputParameter(flattenedTarget)) ||
						(isOutputParameter(flattenedSource) && isOutputParameter(flattenedTarget))) {
					clone.setSource(sourceReference);
					clone.setTarget(targetReference);
				} else {
					throw new RuntimeException("Port members " + flattenedSource.getName() +
							" and " + flattenedTarget.getName() + " in a Binding must have equal direction.");
				}
			} else {
				if (isOutputParameter(flattenedSource) && isInputParameter(flattenedTarget)) {
					clone.setSource(sourceReference);
					clone.setTarget(targetReference);
				} else {
					throw new RuntimeException("Port members " + flattenedSource.getName() +
							" and " + flattenedTarget.getName() + " must be an output and an input port respectively.");
				}
				// This shall only be run for connection usages, not binding
				if (clone.getTarget().getReferencedElement() instanceof final Usage usage &&
						SysMLv2Util.isEvent(usage)) {
					final EqualityExpression equalityExpression = SysMLv2Factory.eINSTANCE.createEqualityExpression();
					final Operator notEqual = SysMLv2Factory.eINSTANCE.createOperator();
					notEqual.setName("!=");
					equalityExpression.setOperator(notEqual);
					final ReferenceExpression callToNext = SysMLv2Factory.eINSTANCE.createReferenceExpression();
					callToNext.setReferencedElement(NEXT);
					callToNext.setInvocation(true);
					final Expression targetclone1 = EcoreUtil.copy(clone.getSource());
					callToNext.getArguments().add(targetclone1);
					equalityExpression.setLeft(callToNext);
					final ReferenceExpression targetclone2 = (ReferenceExpression) EcoreUtil.copy(clone.getSource());
					equalityExpression.setRight(targetclone2);
					clone.setSource(equalityExpression);
				}
			}
			generatedConnections.add(clone);
		});
		return generatedConnections;
	}

	private static List<Pair<DirectedElement, DirectedElement>> buildPortMemberPairs(PortUsage port1, PortUsage port2) {
		final PortDefinition definition1 = (PortDefinition) SysMLv2Util.getType(port1);
		final PortDefinition definition2 = (PortDefinition) SysMLv2Util.getType(port2);
		final var definitionMembers1 = definition1.getMembers();
		final var definitionMembers2 = definition2.getMembers();
		if (definitionMembers1.size() != definitionMembers2.size()) {
			throw new RuntimeException("Port Usages " + port1.getName() + " and " + port2.getName()
			+ " are not compatible as they are typed by Port Definitions containing a different number of members.");
		}
		final List<Pair<DirectedElement, DirectedElement>> memberPairs = new LinkedList<>();
		for (int i = 0; i < definitionMembers1.size(); i++) {
			if (!(definitionMembers1.get(i) instanceof final DirectedElement directedElement1 &&
					definitionMembers2.get(i) instanceof final DirectedElement directedElement2)) {
				throw new RuntimeException("Members at position " + i + " in " + definition1.getName() +
						" and " + definition2.getName() + " are not elements that can have a direction.");
			}
			memberPairs.add(new Pair<DirectedElement, DirectedElement>(directedElement1, directedElement2));
	    }
		for (int i = 0; i < memberPairs.size(); i++) {
			final var pair = memberPairs.get(i);
			final String leftType = SysMLv2Util.getType(pair.getLeft()).getName();
			final String rightType = SysMLv2Util.getType(pair.getRight()).getName();
			if (!leftType.equals(rightType)) {
				throw new RuntimeException("Members at position " + i + " in " + definition1.getName() +
						" and " + definition2.getName() + " have mismatching types: " + leftType + " and " + rightType
						+ " respectively.");
			}
		}
		return memberPairs;
	}

	public enum ComponentsRelationship {
		FIRST_CONTAINS_SECOND, SAME, SECOND_CONTAINS_FIRST;
	}

	public static ComponentsRelationship compareComponents(PartDefinition partDefinition1, PartDefinition partDefinition2) {
		if (typeSelect(partDefinition1.getMembers(), PartUsage.class).stream()
				.filter(partUsage -> partUsage.getTyping() != null &&
						partDefinition2.getName().equals(SysMLv2Util.getType(partUsage).getName()))
				.count() != 0) {
			return ComponentsRelationship.FIRST_CONTAINS_SECOND;
		} else if (typeSelect(partDefinition2.getMembers(), PartUsage.class).stream()
				.filter(partUsage -> partUsage.getTyping() != null &&
						partDefinition1.getName().equals(SysMLv2Util.getType(partUsage).getName()))
				.count() != 0) {
			return ComponentsRelationship.SECOND_CONTAINS_FIRST;
		} else {
			return ComponentsRelationship.SAME;
		}
	}

	public static NamedElement getConnectorEndOwner(Object connectorEnd) {
		if (connectorEnd instanceof final UsageChainExpression chainExpression) {
			return chainExpression.getBody().getReferencedElement();
		} else {
			return null;
		}
	}

	public static PartUsage getDurativeTransitionActivity(Transition transition) {
		final var actFeature = SysMLv2Util.getMemberByName(transition, Usage.class, "act");
		final var activityReference = (ReferenceExpression) actFeature.getValue().getExpression();
		return (PartUsage) activityReference.getReferencedElement();
	}

	public static List<PartUsage> getDurativeTransitionsActivities(PartDefinition component) {
		return typeSelect(ModelUtil.getMissionStateMachine(component).getMembers(), Transition.class).stream()
			.filter(SysMLv2Util::isDurative)
			.map(ModelUtil::getDurativeTransitionActivity)
			.toList();
	}

	private static boolean isOrSpecializesBoundedType(Definition type, String boundedType) {
		if (boundedType.equals(type.getName())) {
			return true;
		} else {
			return type.getSpecialization() != null &&
					type.getSpecialization().getReferences().stream()
					.map(ReferenceExpression::getReferencedElement)
					.anyMatch(element -> element.getName().equals(boundedType));
		}
	}

	public static boolean isIntervalType(Definition type) {
		return isBoundedIntType(type) || isBoundedRealType(type);
	}

	public static boolean isBoundedIntType(Definition type) {
		return isOrSpecializesBoundedType(type, ModelUtil.BOUNDED_INT);
	}

	public static boolean isBoundedRealType(Definition type) {
		return isOrSpecializesBoundedType(type, ModelUtil.BOUNDED_REAL);
	}

	public enum Bound {
		MIN("minValue"), MAX("maxValue");

		public final String name;

		private Bound(String name) {
			this.name = name;
		}
	}

	public static String getBoundValue(Container container, Bound boundType) {
		final Usage bound = SysMLv2Util.getMemberByName(container, Usage.class, boundType.name);
		if (bound == null) {
			if (container instanceof final Usage usage && usage.getTyping() != null) {
				return getBoundValue(SysMLv2Util.getType(usage), boundType);
			} else {
				throw new RuntimeException("Ill-formed bounded type " + (container instanceof NamedElement namedElement ? namedElement.getName() : "") + " : missing " + boundType.name);
			}
		} else {
			return SysMLv2ToSMVExpression.getInstance(Language.CLEANC).serialize(bound.getValue().getExpression());
		}
	}

	public static boolean isExplodtwin(ResourceSet resourceSet) {
		return resourceSet.getResources().stream()
				.map(Resource::getURI)
				.anyMatch(uri -> uri.toString().contains("explodtwin"));
	}

	public static boolean isHavocGeneratedAttribute(Usage usage) {
		return usage.getName().startsWith("generated_") && usage.getName().endsWith("_havoc");
	}

}
