package eu.fbk.eclipse.explodtwin.api.util;

import static java.util.function.Predicate.not;
import static org.eclipse.xtext.EcoreUtil2.eAllContentsAsList;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.EcoreUtil2;

import com.google.common.collect.Lists;
import com.google.common.collect.Streams;

import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.sysmlv2.sysMLv2.ActionUsage;
import eu.fbk.sysmlv2.sysMLv2.AndExpression;
import eu.fbk.sysmlv2.sysMLv2.AssertionAction;
import eu.fbk.sysmlv2.sysMLv2.Assignment;
import eu.fbk.sysmlv2.sysMLv2.AttributeUsage;
import eu.fbk.sysmlv2.sysMLv2.Binding;
import eu.fbk.sysmlv2.sysMLv2.BooleanLiteral;
import eu.fbk.sysmlv2.sysMLv2.Calculation;
import eu.fbk.sysmlv2.sysMLv2.CalculationDefinition;
import eu.fbk.sysmlv2.sysMLv2.ConnectionElement;
import eu.fbk.sysmlv2.sysMLv2.ConnectionUsage;
import eu.fbk.sysmlv2.sysMLv2.ConstraintUsage;
import eu.fbk.sysmlv2.sysMLv2.Definition;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Direction;
import eu.fbk.sysmlv2.sysMLv2.Effect;
import eu.fbk.sysmlv2.sysMLv2.EqualityExpression;
import eu.fbk.sysmlv2.sysMLv2.Expression;
import eu.fbk.sysmlv2.sysMLv2.Guard;
import eu.fbk.sysmlv2.sysMLv2.ItemDefinition;
import eu.fbk.sysmlv2.sysMLv2.ItemUsage;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.Operator;
import eu.fbk.sysmlv2.sysMLv2.OperatorExpression;
import eu.fbk.sysmlv2.sysMLv2.Package;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.Payload;
import eu.fbk.sysmlv2.sysMLv2.PerformActionReference;
import eu.fbk.sysmlv2.sysMLv2.PortDefinition;
import eu.fbk.sysmlv2.sysMLv2.PortUsage;
import eu.fbk.sysmlv2.sysMLv2.QualifiedNameExpression;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.SendUsage;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.SuccessionNode;
import eu.fbk.sysmlv2.sysMLv2.SysMLv2Factory;
import eu.fbk.sysmlv2.sysMLv2.TernaryOperatorExpression;
import eu.fbk.sysmlv2.sysMLv2.Transition;
import eu.fbk.sysmlv2.sysMLv2.TriggerAction;
import eu.fbk.sysmlv2.sysMLv2.TypedElement;
import eu.fbk.sysmlv2.sysMLv2.Typing;
import eu.fbk.sysmlv2.sysMLv2.UsageChainExpression;
import eu.fbk.sysmlv2.sysMLv2.UsageReferenceExpression;
import eu.fbk.sysmlv2.util.SysMLv2Util;

public class PostProcessor {

	private final ResourceSet resourceSet;
	private ItemDefinition eventDefinition = null;
	private PartDefinition activityDefinition = null;
	private PortDefinition conjugatedActivityPerformedPort = null;
	private ItemDefinition activityPerformed = null;
	private final boolean isExplodtwin;

	public PostProcessor(ResourceSet resourceSet, boolean isExplodtwin) {
		this.resourceSet = resourceSet;
		this.isExplodtwin = isExplodtwin;
		if (isExplodtwin) {
			eventDefinition = getFromIndex(ItemDefinition.class, "Event");
			activityDefinition = getFromIndex(PartDefinition.class, "Activity");
			conjugatedActivityPerformedPort = getFromIndex(PortDefinition.class, "~ActivityPerformedPort");
			activityPerformed = getFromIndex(ItemDefinition.class, "ActivityPerformed");
			ModelUtil.getPortUsagesInPart(activityDefinition).forEach(portUsage ->
				typeSelect(SysMLv2Util.getType(portUsage).getMembers(), DirectedElement.class)
					.forEach(element -> ModelUtil.flattenNestedMember(element, portUsage)));
		}
	}

	private <T extends NamedElement> T getFromIndex(Class<T> type, String name) {
		return Streams.stream(resourceSet.getAllContents())
			.filter(type::isInstance).map(type::cast)
			.filter(namedElement -> EcoreUtil2.getContainerOfType(namedElement, Package.class).isLibrary() &&
					name.equals(namedElement.getName()))
			.findAny().orElseThrow(() -> new RuntimeException("Could not retrieve " + type.getSimpleName() + " " + name));
	}

	private void process(Resource resource) {
		installAdditionalEventMembers(resource);
		installStutterTransitions(resource);
		installActivities(resource);
		flattenPorts(resource);
		installDurativeTransitions(resource);
		installDanglingPorts(resource);
	}

	/*
	 * The purpose of this post-processing step is to trigger composite ports flattening once, for each
	 * component and from the same "entry point" (e.g. ModelUtil.getNonStaticPorts(Part)) before the
	 * translation process begins, in order for the flattening order to be fixed no matter which
	 * translation command is run.
	 */
	public void flattenPorts(Resource resource) {
		if (!SysMLv2Util.resourceBelongsToLibrary(resource)) {
			typeSelect(eAllContentsAsList(resource), PartDefinition.class).forEach(ModelUtil::getNonStaticPorts);
		}
	}

	public void performPostProcessing() {
		if (!isExplodtwin) {
			return;
		}
		final List<Resource> libraryResources = new ArrayList<>(resourceSet.getResources());
		libraryResources.removeIf(not(SysMLv2Util::resourceBelongsToLibrary));
		final List<Resource> modelResources = new ArrayList<>(resourceSet.getResources());
		modelResources.removeIf(SysMLv2Util::resourceBelongsToLibrary);
		libraryResources.forEach(this::process);
		modelResources.forEach(this::process);
		/*
		 * Install setter transitions after the rest of the processing actions have been
		 * performed, to ensure implicit event members have been added to all PortDefinitions.
		 */
		modelResources.forEach(this::installSetterTransitions);
	}

	private static Transition createLoopTransition(StateUsage state) {
		final Transition transition = SysMLv2Factory.eINSTANCE.createTransition();
		transition.setSource(state);
		transition.setDestination(state);
		return transition;
	}

	private static void createTypingForUsage(TypedElement usageOrPayload, Definition definition) {
		final Typing typing = SysMLv2Factory.eINSTANCE.createTyping();
		final QualifiedNameExpression expression = SysMLv2Factory.eINSTANCE.createQualifiedNameExpression();
		expression.setReferencedElement(definition);
		typing.getReferences().add(expression);
		usageOrPayload.setTyping(typing);
	}

	private void installAdditionalEventMembers(Resource resource) {
		typeSelect(eAllContentsAsList(resource), SendUsage.class).forEach(sendUsage -> {
			if (sendUsage.getPayload() instanceof final ReferenceExpression reference &&
					reference.getReferencedElement() instanceof ItemDefinition) {
				return;
			}
			if (sendUsage.getTransport() == null) {
				return;
			}
			if (!(sendUsage.getTransport().getReferencedElement() instanceof final PortUsage transport)) {
				return;
			}
			PortDefinition portDefinition = (PortDefinition) SysMLv2Util.getType(transport);
			final boolean installOutputEventPort =
					typeSelect(portDefinition.getMembers(), ItemUsage.class).stream()
							.noneMatch(SysMLv2Util::itemUsageIsEvent);
			final boolean installInputEventPort =
					typeSelect(portDefinition.getConjugatedPortDefinition().getMembers(), ItemUsage.class).stream()
							.noneMatch(SysMLv2Util::itemUsageIsEvent);
			if (installOutputEventPort) {
				portDefinition.getMembers().add(generateEventPort(Direction.OUT));
			}
			if (installInputEventPort) {
				portDefinition.getConjugatedPortDefinition().getMembers().add(generateEventPort(Direction.IN));
			}
		});
	}

	private ItemUsage generateEventPort(Direction direction) {
		final ItemUsage eventPort = SysMLv2Factory.eINSTANCE.createItemUsage();
		eventPort.setName("generatedEventPort");
		eventPort.setDirection(direction);
		createTypingForUsage(eventPort, eventDefinition);
		return eventPort;
	}

	private void installStutterTransitions(Resource resource) {
		typeSelect(eAllContentsAsList(resource), StateUsage.class).stream()
			.filter(stateUsage -> SysMLv2Util.isStateMachine(stateUsage) &&
					!"Environment".equals(SysMLv2Util.getSpecializedDefinitionName(
							EcoreUtil2.getContainerOfType(stateUsage, PartDefinition.class))))
			.forEach(stateMachine -> typeSelect(stateMachine.getMembers(), StateUsage.class).stream()
					.map(PostProcessor::createLoopTransition)
					.forEach(transition -> {
						transition.setStutter(true);
						stateMachine.getMembers().add(transition);
					}));
	}

	private void installActivities(Resource resource) {
		final StateUsage activityStateMachine = typeSelect(activityDefinition.getMembers(), StateUsage.class).get(0);
		final List<PortUsage> portUsages = typeSelect(activityDefinition.getMembers(), PortUsage.class);
		final List<AttributeUsage> attributeUsages = typeSelect(activityDefinition.getMembers(), AttributeUsage.class);
		typeSelect(eAllContentsAsList(resource), PartDefinition.class).stream()
			.filter(SysMLv2Util::partDefinitionIsActivity)
			.forEach(partDefinition -> {
				final StateUsage stateMachine = EcoreUtil.copy(activityStateMachine);
				stateMachine.setName(stateMachine.getName() + "_" + partDefinition.getName());
				partDefinition.getMembers().add(stateMachine);
				final List<PortUsage> portsClones = portUsages.stream()
					.map(EcoreUtil::copy)
					.toList();
				partDefinition.getMembers().addAll(portsClones);
				attributeUsages.stream()
					.map(EcoreUtil::copy)
					.forEach(partDefinition.getMembers()::add);
				// The state machine and port usages have now been copied and transfered.
				EcoreUtil2.getAllContentsOfType(stateMachine, ReferenceExpression.class).forEach(reference -> {
					if (reference.getReferencedElement() instanceof final PortUsage portUsage) {
						final String name = portUsage.getName();
						final PortUsage clone = portsClones.stream()
								.filter(portClone -> portClone.getName().equals(name)).findFirst().orElse(null);
						if (clone == null) { // Should never happen in practice, unless the user fiddles with the libraries.
							return;
						}
						reference.setReferencedElement(clone);
						EcoreUtil2.typeSelect(Lists.newArrayList(resource.getAllContents()), ReferenceExpression.class).forEach(reference2 -> {
							if (reference2.getReferencedElement() instanceof final PortUsage portUsage2
									&& portUsage == portUsage2) {
								reference2.setReferencedElement(clone);
							}
						});
					} else if (reference.getReferencedElement() instanceof final Calculation calculation) {
						final String name = calculation.getName();
						final Calculation redefinedCalculation = SysMLv2Util.getMemberByName(partDefinition, Calculation.class, name);
						if (redefinedCalculation == null) {
							return;
						}
						reference.setReferencedElement(redefinedCalculation);
					}
				});
				EcoreUtil2.getAllContentsOfType(stateMachine, PerformActionReference.class).forEach(performAction -> {
					final String name = performAction.getReference().getReferencedElement().getName();
					final ActionUsage redefinedAction = SysMLv2Util.getMemberByName(partDefinition, ActionUsage.class, name);
					if (redefinedAction == null) {
						return;
					}
					performAction.getReference().setReferencedElement(redefinedAction);
				});
				List.of("precondition", "postcondition").forEach(constraintName -> {
					if (SysMLv2Util.getMemberByName(partDefinition, ConstraintUsage.class, constraintName) == null) {
						final var stub = SysMLv2Util.getMemberByName(activityDefinition, ConstraintUsage.class, constraintName);
						final var nonAbstractStub = EcoreUtil2.copy(stub);
						nonAbstractStub.setAbstract(false);
						partDefinition.getMembers().add(nonAbstractStub);
					}
				});
				final ConstraintUsage libraryInvariant = SysMLv2Util.getMemberByName(activityDefinition, ConstraintUsage.class, "invariant");
				Expression expression = EcoreUtil.copy(libraryInvariant.getExpression());
				var invariant = SysMLv2Util.getMemberByName(partDefinition, ConstraintUsage.class, "invariant");
				if (invariant != null) {
					final Expression otherExpression = invariant.getExpression();
					final AndExpression andExpression = SysMLv2Factory.eINSTANCE.createAndExpression();
					andExpression.setLeft((OperatorExpression) expression);
					andExpression.setRight((OperatorExpression) otherExpression);
					final Operator andOperator = SysMLv2Factory.eINSTANCE.createOperator();
					andOperator.setName("and");
					andExpression.setOperator(andOperator);
					invariant.setExpression(andExpression);
				} else {
					invariant = SysMLv2Factory.eINSTANCE.createConstraintUsage();
					invariant.setExpression(expression);
				}
				final StateUsage progressState = SysMLv2Util.getMemberByName(stateMachine, StateUsage.class, "Progress");
				final AssertionAction assertionAction = SysMLv2Factory.eINSTANCE.createAssertionAction();
				assertionAction.setConstraint(invariant);
				progressState.getMembers().add(assertionAction);
			});
	}

	private void installDurativeTransitions(Resource resource) {
		typeSelect(eAllContentsAsList(resource), Transition.class).stream().filter(SysMLv2Util::isDurative)
			.forEach(durativeTransition -> {
				final PartUsage activity = ModelUtil.getDurativeTransitionActivity(durativeTransition);
				final PartDefinition subsystem = EcoreUtil2.getContainerOfType(durativeTransition, PartDefinition.class);
				if (SysMLv2Util.getMemberByName(subsystem, PortUsage.class, activity.getName() + "_done_input") != null) {
					return;
				}
				final PortUsage activityPerformedInputPort = SysMLv2Factory.eINSTANCE.createPortUsage();
				createTypingForUsage(activityPerformedInputPort, conjugatedActivityPerformedPort);
				activityPerformedInputPort.setName(activity.getName() + "_done_input");
				subsystem.getMembers().add(activityPerformedInputPort);
				final ConnectionUsage connection = SysMLv2Factory.eINSTANCE.createConnectionUsage();
				final UsageChainExpression output = SysMLv2Factory.eINSTANCE.createUsageChainExpression();
				final UsageReferenceExpression outputBody = SysMLv2Factory.eINSTANCE.createUsageReferenceExpression();
				outputBody.setReferencedElement(activity);
				output.setBody(outputBody);
				final PortUsage referenceTarget = SysMLv2Util.getMemberByName(activityDefinition, PortUsage.class, "activityPerformedPort");
				output.setReferencedElement(referenceTarget);
				connection.setSource(output);
				final QualifiedNameExpression input = SysMLv2Factory.eINSTANCE.createQualifiedNameExpression();
				input.setReferencedElement(activityPerformedInputPort);
				connection.setTarget(input);
				subsystem.getMembers().add(connection);

				final Payload payload = SysMLv2Factory.eINSTANCE.createPayload();
				createTypingForUsage(payload, activityPerformed);
				final TriggerAction trigger = SysMLv2Factory.eINSTANCE.createTriggerAction();
				trigger.setPayload(payload);

				final QualifiedNameExpression referenceToInputPort = SysMLv2Factory.eINSTANCE.createQualifiedNameExpression();
				referenceToInputPort.setReferencedElement(activityPerformedInputPort);
				trigger.setPort(referenceToInputPort);
				durativeTransition.getTriggers().add(trigger);
			});
	}

	private void installSetterTransitions(Resource resource) {
		if (SysMLv2Util.resourceBelongsToLibrary(resource)) {
			return;
		}
		typeSelect(eAllContentsAsList(resource), PartUsage.class).forEach(this::installSetterTransitionsForComponentInstance);
	}

	private static PartUsage getSetterSender(PartUsage receivingInstance, PortUsage receivingPort) {
		final List<ConnectionElement> allConnections = ModelUtil.getAllContentsOfTypeFromModel(
				EcoreUtil2.getResourceSet(receivingInstance), ConnectionElement.class);
		PortUsage sendingPort = receivingPort;
		PortUsage tmp = null;
		Pair<PortUsage, ConnectionElement> pair = null;
		ConnectionElement lastConnection = null;
		while (sendingPort != tmp) {
			tmp = sendingPort;
			pair = findConnectionAndEnd(sendingPort, receivingInstance, allConnections);
			sendingPort = pair.getLeft();
			if (pair.getRight() != null) {
				lastConnection = pair.getRight();
			}
		}
		boolean landedOnSource = true;
		if (lastConnection.getTarget().getReferencedElement() == sendingPort) {
			landedOnSource = false;
		}
		return (PartUsage) ModelUtil.getConnectorEndOwner(landedOnSource ? lastConnection.getSource() : lastConnection.getTarget());
	}

	private void installSettersIntoTransitions(List<ActionUsage> setters, List<Transition> transitions) {
		transitions.forEach(transition -> {
			final Effect effect = SysMLv2Factory.eINSTANCE.createEffect();
			final ActionUsage effectAction = SysMLv2Factory.eINSTANCE.createActionUsage();
			for (int i = 0; i < setters.size(); i++) {
				final ActionUsage setter = setters.get(i);
				final ActionUsage action = (ActionUsage) setter.getMembers().get(0);
				final TriggerAction trigger = (TriggerAction) action.getMembers().get(0);
				final PortUsage port = (PortUsage) trigger.getPort().getReferencedElement();
				final TriggerAction transitionTrigger = SysMLv2Factory.eINSTANCE.createTriggerAction();
				final Payload payload = SysMLv2Factory.eINSTANCE.createPayload();
				createTypingForUsage(payload, eventDefinition);
				transitionTrigger.setPayload(payload);
				final UsageChainExpression portReference = SysMLv2Factory.eINSTANCE.createUsageChainExpression();
				portReference.setReferencedElement(port);
				transitionTrigger.setPort(portReference);
				transition.getTriggers().add(transitionTrigger);
				if (!(setter.getMembers().get(1) instanceof final SuccessionNode successionNode &&
						successionNode.getNode() instanceof final Assignment setterAssignment)) {
					return;
				}
				final Assignment assignment = EcoreUtil.copy(setterAssignment);
				if (!(assignment.getValue().getExpression() instanceof final UsageChainExpression usageChainExpression)) {
					return;
				}
				usageChainExpression.getBody().setReferencedElement(port);
				usageChainExpression.setReferencedElement(SysMLv2Util.getDataMemberFromPort(port));
				final TernaryOperatorExpression ternaryOperatorExpression = SysMLv2Factory.eINSTANCE.createTernaryOperatorExpression();
				final Guard guard = SysMLv2Factory.eINSTANCE.createGuard();
				final EqualityExpression eventOccurredExpression = SysMLv2Factory.eINSTANCE.createEqualityExpression();
				final UsageChainExpression referenceToEventMember = SysMLv2Factory.eINSTANCE.createUsageChainExpression();
				final UsageReferenceExpression body = SysMLv2Factory.eINSTANCE.createUsageReferenceExpression();
				body.setReferencedElement(port);
				referenceToEventMember.setBody(body);
				referenceToEventMember.setReferencedElement(SysMLv2Util.getEventMemberFromPort(port));
				eventOccurredExpression.setLeft(referenceToEventMember);
				final Operator operator = SysMLv2Factory.eINSTANCE.createOperator();
				operator.setName("==");
				eventOccurredExpression.setOperator(operator);
				final BooleanLiteral trueLiteral = SysMLv2Factory.eINSTANCE.createBooleanLiteral();
				trueLiteral.setValue("true");
				eventOccurredExpression.setRight(trueLiteral);
				guard.setExpression(eventOccurredExpression);
				ternaryOperatorExpression.setGuard(guard);
				final ReferenceExpression callToNext = SysMLv2Factory.eINSTANCE.createReferenceExpression();
				callToNext.setReferencedElement(ModelUtil.NEXT);
				callToNext.setInvocation(true);
				callToNext.getArguments().add(usageChainExpression);
				ternaryOperatorExpression.setLeft(callToNext);
				final Operator elseOperator = SysMLv2Factory.eINSTANCE.createOperator();
				elseOperator.setName("else");
				ternaryOperatorExpression.setOperator(elseOperator);
				final UsageReferenceExpression referenceToTargetVariable = SysMLv2Factory.eINSTANCE.createUsageReferenceExpression();
				referenceToTargetVariable.setReferencedElement(setterAssignment.getElement().getReferencedElement());
				ternaryOperatorExpression.setRight(referenceToTargetVariable);
				assignment.getValue().setExpression(ternaryOperatorExpression);
				effectAction.getMembers().add(assignment);
			}
			effect.setAction(effectAction);
			transition.setEffect(effect);
		});

	}

	private static PortUsage getSetterActionPortUsage(ActionUsage setterAction) {
		final List<ActionUsage> actionMembers = typeSelect(setterAction.getMembers(), ActionUsage.class);
		if (actionMembers.isEmpty()) {
			throw new RuntimeException("No trigger in Setter action " + setterAction.getName() +
					", inside component " + EcoreUtil2.getContainerOfType(setterAction, PartDefinition.class).getName());
		}
		final ActionUsage action = actionMembers.get(0);
		if (!(action.getMembers().get(0) instanceof final TriggerAction trigger)) {
			throw new RuntimeException("Ill-formed trigger action in Setter " + setterAction.getName() +
					". No 'accept' action found.");
		}
		final PortUsage portUsage = (PortUsage) trigger.getPort().getReferencedElement();
		return portUsage;
	}

	private void installSetterTransitionsForComponentInstance(PartUsage receivingInstance) {
		final PartDefinition component = ModelUtil.getComponentType(receivingInstance);
		/*
		 * For the component above, we compute a map that groups all the component's
		 * Setter actions by their sending component instance.
		 */
		final Map<PartUsage, List<ActionUsage>> settersSendersMap = new HashMap<>();
		typeSelect(component.getMembers(), ActionUsage.class).stream()
			.filter(SysMLv2Util::isSetterAction)
			.forEach(setter -> {
				final PortUsage portUsage = getSetterActionPortUsage(setter);
				final PartUsage sender = getSetterSender(receivingInstance, portUsage);
				settersSendersMap.putIfAbsent(sender, new ArrayList<>());
				settersSendersMap.get(sender).add(setter);
			});
		/*
		 * The map has been populated. Now we search for the state machine we
		 * will install the setter transitions into. For Subsystems, this is
		 * the mission state machine, which at the moment is a Subsystem's only
		 * state machine. For the Environment, this is its standard state machine.
		 */
		final List<StateUsage> stateMachines = typeSelect(component.getMembers(), StateUsage.class);
		if (stateMachines.isEmpty()) {
			return;
		}
		final StateUsage targetStateMachine = stateMachines.get(0);
		/*
		 * We need to install one setter transition (i.e. a self loop) for each
		 * state in the state machine, for each sending component.
		 */
		final List<StateUsage> states = typeSelect(targetStateMachine.getMembers(), StateUsage.class);
		settersSendersMap.entrySet().stream()
			.sorted(Comparator.comparing(entry -> entry.getKey().getName()))
			.forEach(entry -> {
				final PartUsage sender = entry.getKey();
				final List<ActionUsage> actions = entry.getValue();
				/*
				 * If the component is a Subsystem and the sender is an Activity of its,
				 * the setter transition is not installed, and instead its post-effect is squashed
				 * onto the existing durative transition for that Activity, which is assumed
				 * to exist (if branch). A setter loop is created for pre-effects.
				 * Otherwise, we create a dedicated setter loop transition (else branch).
				 */
				if (ModelUtil.isActivity(sender) && component.getMembers().contains(sender)) {
					// Install setters for post-effects into durative transitions
					final List<Transition> durativeTransitions = typeSelect(targetStateMachine.getMembers(), Transition.class).stream()
							.filter(SysMLv2Util::isDurative)
							.filter(transition -> ModelUtil.getDurativeTransitionActivity(transition) == sender)
							.toList();
					installSettersIntoTransitions(actions, durativeTransitions);

					// Install setters for pre-effects into setter loop transitions
					final var transitionsSourceStates = durativeTransitions.stream()
						.map(Transition::getSource)
						.distinct()
						.toList();

					final List<Transition> loops = new ArrayList<>();

					final Binding binding = EcoreUtil2.getAllContentsOfType(component, UsageChainExpression.class).stream()
						.filter(usageChain -> EcoreUtil.equals(usageChain.getBody().getReferencedElement(), sender) &&
								usageChain.getReferencedElement().getName().equals("startActivityPort"))
						.map(usageChain -> EcoreUtil2.getContainerOfType(usageChain, Binding.class))
						.findFirst().orElseThrow(() -> new RuntimeException("No binding for startActivityPort"));
					final ReferenceExpression reference;
					if (binding.getTarget().getReferencedElement().getName().equals("startActivityPort")) {
						reference = (ReferenceExpression) binding.getSource();
					} else {
						reference = binding.getTarget();
					}
					final PortUsage port = (PortUsage) reference.getReferencedElement();
					final TriggerAction transitionTrigger = SysMLv2Factory.eINSTANCE.createTriggerAction();
					final Payload payload = SysMLv2Factory.eINSTANCE.createPayload();
					final var eventMember = SysMLv2Util.getEventMemberFromPort(port);
					final var eventType = SysMLv2Util.getType(eventMember);
					createTypingForUsage(payload, eventType);
					transitionTrigger.setPayload(payload);
					final UsageChainExpression portReference = SysMLv2Factory.eINSTANCE.createUsageChainExpression();
					portReference.setReferencedElement(port);
					transitionTrigger.setPort(portReference);

					transitionsSourceStates.forEach(state -> {
						final Transition loop = createLoopTransition(state);
						loop.setName(state.getName() + "_preeffects_of_" + sender.getName());
						loop.getTriggers().add(transitionTrigger);
						loops.add(loop);
					});
					installSettersIntoTransitions(actions, loops);
					targetStateMachine.getMembers().addAll(loops);
				} else {
					final var loops = states.stream().map(PostProcessor::createLoopTransition).toList();
					loops.forEach(loop -> loop.setName(loop.getSource().getName() + "_setter_from_" + sender.getName()));
					installSettersIntoTransitions(actions, loops);
					targetStateMachine.getMembers().addAll(loops);
				}
			});
	}

	private static Pair<PortUsage, ConnectionElement> findConnectionAndEnd(PortUsage portUsage, PartUsage owningComponentInstance, List<ConnectionElement> connections) {
		return List.copyOf(connections).stream()
				.<Pair<PortUsage, ConnectionElement>>mapMulti((connection, acceptor) -> {
					if (connection.getSource() instanceof final ReferenceExpression sourceReference &&
							sourceReference.getReferencedElement() == portUsage &&
							(ModelUtil.getConnectorEndOwner(connection.getSource()) == null ||
							ModelUtil.getConnectorEndOwner(connection.getSource()) == owningComponentInstance)) {
						connections.remove(connection);
						final var rawTarget = (PortUsage) connection.getTarget().getReferencedElement();
						acceptor.accept(new Pair<PortUsage, ConnectionElement>(rawTarget, connection));
					} else if (connection.getTarget().getReferencedElement() == portUsage &&
							(ModelUtil.getConnectorEndOwner(connection.getTarget()) == null ||
							ModelUtil.getConnectorEndOwner(connection.getTarget()) == owningComponentInstance)) {
						connections.remove(connection);
						final var rawSource = (PortUsage) ((ReferenceExpression) connection.getSource()).getReferencedElement();
						acceptor.accept(new Pair<PortUsage, ConnectionElement>(rawSource, connection));
					}
				})
				.findAny().orElse(new Pair<PortUsage, ConnectionElement>(portUsage, null));
	}

	private void installDanglingPorts(Resource resource) {
		typeSelect(eAllContentsAsList(resource), ReferenceExpression.class).stream()
			.filter(reference -> reference.getReferencedElement() instanceof final CalculationDefinition calculation
					&& calculation.getName().equals("havoc"))
			.forEach(reference -> {
				final PartDefinition partDefinition = EcoreUtil2.getContainerOfType(reference, PartDefinition.class);
				final SendUsage sendUsage = EcoreUtil2.getContainerOfType(reference, SendUsage.class);
				final PortUsage sendUsagePort = (PortUsage) sendUsage.getTransport().getReferencedElement();
				final var dataMember = SysMLv2Util.getDataMemberFromPort(sendUsagePort);
				final String assignedElementName = sendUsagePort.getName().substring(4);
				final AttributeUsage havocPort = SysMLv2Factory.eINSTANCE.createAttributeUsage();
				havocPort.setName("generated_" + assignedElementName + "_havoc");
				final Typing typing = EcoreUtil.copy(dataMember.getTyping());
				havocPort.setTyping(typing);
				havocPort.setDirection(Direction.IN);
				partDefinition.getMembers().add(havocPort);
				reference.setReferencedElement(havocPort);
			});
	}

}
