/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.api.util;

import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.util.EcoreUtil;

import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.sysmlv2.sysMLv2.BooleanLiteral;
import eu.fbk.sysmlv2.sysMLv2.BoundedExpression;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.EmptyExpression;
import eu.fbk.sysmlv2.sysMLv2.EnumUsage;
import eu.fbk.sysmlv2.sysMLv2.Expression;
import eu.fbk.sysmlv2.sysMLv2.ListExpression;
import eu.fbk.sysmlv2.sysMLv2.Literal;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.OperatorExpression;
import eu.fbk.sysmlv2.sysMLv2.PortUsage;
import eu.fbk.sysmlv2.sysMLv2.PrimaryExpression;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.TernaryOperatorExpression;
import eu.fbk.sysmlv2.sysMLv2.UnevaluatedExpression;
import eu.fbk.sysmlv2.sysMLv2.UsageChainExpression;
import eu.fbk.sysmlv2.sysMLv2.UsageExpression;
import eu.fbk.sysmlv2.sysMLv2.UsageReferenceExpression;

public class SysMLv2ToSMVExpression {

	/**
	 * Language to be used in the serialization.
	 */
	public static enum Language {
		CLEANC, LTL, LTLSMV;

		public static final String LTL_STRING = "OCRA";
		public static final String CLEANC_STRING = "CleanC";
		public static final String LTLSMV_STRING = "LTLSMV";

		public static Language fromString(String value) {
			return switch (value) {
			case LTL_STRING -> LTL;
			case CLEANC_STRING -> CLEANC;
			case LTLSMV_STRING -> LTLSMV;
			default -> throw new IllegalArgumentException("Unexpected value for language: " + value);
			};
		}
	}

	private final Language language;

	private boolean mapEnumsToInts = true;

	private static final SysMLv2ToSMVExpression CLEANC_INSTANCE = new SysMLv2ToSMVExpression(Language.CLEANC);
	private static final SysMLv2ToSMVExpression LTL_INSTANCE = new SysMLv2ToSMVExpression(Language.LTL);
	private static final SysMLv2ToSMVExpression LTLSMV_INSTANCE = new SysMLv2ToSMVExpression(Language.LTLSMV);
	private static final SysMLv2ToSMVExpression LTLSMV_INSTANCE_WITHOUT_ENUM_MAPPING = new SysMLv2ToSMVExpression(Language.LTLSMV);

	static {
		LTLSMV_INSTANCE_WITHOUT_ENUM_MAPPING.mapEnumsToInts = false;
	}

	private static final Set<String> BINARY_LTL_OPERATORS = Set.of(
		"iff", "until", "U", "releases", "V", "since", "S", "triggered", "T", "at_next", "at_last");

	/**
	 * Creates a serializer for the given language.
	 * @param language
	 */
	private SysMLv2ToSMVExpression(Language language) {
		this.language = language;
	}

	public static SysMLv2ToSMVExpression getInstance(Language language) {
		return switch (language) {
		case CLEANC -> CLEANC_INSTANCE;
		case LTL -> LTL_INSTANCE;
		case LTLSMV -> LTLSMV_INSTANCE;
		};
	}

	public static SysMLv2ToSMVExpression getInstance(String language) {
		return getInstance(Language.fromString(language));
	}

	public static SysMLv2ToSMVExpression getLTLSMVInstanceWithoutEnumMapping() {
		return LTLSMV_INSTANCE_WITHOUT_ENUM_MAPPING;
	}

	private String serialize(UsageChainExpression expression) {
		final NamedElement referencedElement = expression.getReferencedElement();
		if (referencedElement instanceof final DirectedElement referencedDirectedElement &&
				expression.getBody().getReferencedElement() instanceof final PortUsage portUsage) {
			final DirectedElement flattened = ModelUtil.NESTED_TO_FLATTENED.get(new Pair<>(referencedDirectedElement, portUsage));
			if (flattened != null) {
				final var clone = EcoreUtil.copy(expression.getBody());
				clone.setReferencedElement(flattened);
				return serialize(clone);
			}
		}
		return new StringBuilder()
			.append(serialize(expression.getBody()))
			.append(".")
			.append(expression.getReferencedElement().getName())
			.toString();
	}

	private String serialize(UsageExpression expression) {
		if (expression instanceof final UsageReferenceExpression ure) {
			return ure.getReferencedElement().getName();
		} else if (expression instanceof final UsageChainExpression uce) {
			return serialize(uce);
		} else {
			return null;
		}
	}

	private String serialize(PrimaryExpression expression) {
		if (expression instanceof final Literal literal) {
			return serialize(literal);
		} else if (expression instanceof final ReferenceExpression referenceExpression) {
			return serialize(referenceExpression);
		} else if (expression instanceof final BoundedExpression boundedExpression) {
			return serialize(boundedExpression);
		} else if (expression instanceof final ListExpression listExpression) {
			return serialize(listExpression);
		} else {
			return null;
		}
	}

	private String serialize(OperatorExpression expression) {
		if (expression instanceof final PrimaryExpression primaryExpression) {
			return serialize(primaryExpression);
		} else if (expression instanceof EmptyExpression) {
			return "";
		} else if (expression.getLeft() instanceof EmptyExpression) {
			return serializeUnaryExpression(expression);
		} else if (expression instanceof final TernaryOperatorExpression ternaryOperatorExpression) {
			return serialize(ternaryOperatorExpression);
		} else {
			return serializeBinaryExpression(expression);
		}
	}

	private String serialize(TernaryOperatorExpression expression) {
		return new StringBuilder()
			.append("(").append(serialize(expression.getGuard().getExpression())).append(")")
			.append(translateOperator(expression))
			.append("(").append(serialize(expression.getLeft())).append(")")
			.append(" : ")
			.append("(").append(serialize(expression.getRight())).append(")")
			.toString();
	}

	private String serializeUnaryExpression(OperatorExpression expression) {
		return new StringBuilder()
			.append(translateOperator(expression))
			.append(serialize(expression.getRight()))
			.toString();
	}

	private String serializeBinaryExpression(OperatorExpression expression) {
		final StringBuilder buffer = new StringBuilder();
		final String operator = translateOperator(expression);
		final String leftExpression = serialize(expression.getLeft());
		final String rightExpression = serialize(expression.getRight());
		/*
		 * Implications are not supported by CleanC, even though SMV supports them.
		 * Thus, we translate 'p implies q' as '!p || q' when the target language
		 * is CleanC. This will then be translated into '!p | q', which does not
		 * make use of the SMV '->' operator but is still equivalent.
		 */
		if (" -> ".equals(operator) && language == Language.CLEANC) {
			final String conjugatedAntecedent = "(!(" + leftExpression + "))";
			final String consequent = rightExpression;
			return conjugatedAntecedent + " || " + consequent;
		} else if (" ^ ".equals(operator)) {
			return String.format("pow(%s,%s)", leftExpression, rightExpression);
		} else {
			buffer.append("( ")
				.append(leftExpression).append(operator).append(rightExpression)
				.append(" )");
			return buffer.toString();
		}
	}

	private String serialize(UnevaluatedExpression expression) {
		return serialize(expression.getExpression());
	}

	private String serialize(Literal expression) {
		return expression instanceof final BooleanLiteral booleanLiteral ? serialize(booleanLiteral) : expression.getValue();
	}

	private String serialize(BooleanLiteral booleanLiteral) {
		return language == Language.CLEANC ? booleanLiteral.getValue() : booleanLiteral.getValue().toUpperCase();
	}

	private String serialize(ReferenceExpression referenceExpression) {
		final StringBuilder buffer = new StringBuilder();
		final String calledElementName = referenceExpression.getReferencedElement().getName();
		if (BINARY_LTL_OPERATORS.contains(calledElementName)) {
			return buffer.append("(" + serialize(referenceExpression.getArguments().get(0)))
				.append(" " + calledElementName.replace("_", " ") + " ")
				.append(serialize(referenceExpression.getArguments().get(1)) + " )")
				.toString();
		}
		if (mapEnumsToInts && referenceExpression.getReferencedElement() instanceof final EnumUsage enumUsage
				&& enumUsage.getValue() != null) {
			return buffer.append(serialize(enumUsage.getValue().getExpression())).toString();
		}
		if (referenceExpression instanceof final UsageChainExpression usageChainExpression) {
			buffer.append(serialize(usageChainExpression));
		} else {
			buffer.append(transformLTLOperatorNames(calledElementName));
		}
		if (referenceExpression.isInvocation() && !referenceExpression.getArguments().isEmpty()) {
			buffer.append(serializeArguments(referenceExpression));
		}
		return buffer.toString();
	}

	private String transformLTLOperatorNames(String name) {
		return switch (name) {
		case "always" -> language == Language.LTL ? " always " : " G ";
		case "in_the_future" -> language == Language.LTL ? " in the future " : " F ";
		case "then" -> language == Language.LTL ? " then " : " X ";
		case "in_the_past" -> language == Language.LTL ? " in the past " : " O ";
		case "historically" -> language == Language.LTL ? " historically " : " H ";
		case "previously" -> language == Language.LTL ? " previously " : " Y ";
		case "until" -> language == Language.LTL ? " until " : " U ";
		case "releases" -> language == Language.LTL ? " releases " : " V ";
		case "since" -> language == Language.LTL ? " since " : " S ";
		case "triggered" -> language == Language.LTL ? " triggered " : " T ";
		default -> name;
		};
	}

	private String serializeArguments(ReferenceExpression expression) {
		return new StringBuilder()
			.append("(")
			.append(expression.getArguments().stream().map(this::serialize).collect(Collectors.joining(",")))
			.append(")")
			.toString();
	}

	private String serialize(BoundedExpression expression) {
		return new StringBuilder()
			.append("(").append(serialize(expression.getExpression())).append(")")
			.toString();
	}

	private String serialize(ListExpression expression) {
		return new StringBuilder()
			.append("(")
			.append(expression.getExpressions().stream().map(this::serialize).collect(Collectors.joining(",")))
			.append(")")
			.toString();
	}

	public String serialize(Expression expression) {
		if (expression instanceof final OperatorExpression operatorExpression) {
			return serialize(operatorExpression);
		} else if (expression instanceof final UnevaluatedExpression unevaluatedExpression) {
			return serialize(unevaluatedExpression);
		} else {
			return null;
		}
	}

	private String translateOperator(OperatorExpression expression) {
		final String operatorString = expression.getOperator().getName().trim();
		return switch (operatorString) {
		case "not" -> "!";
		case "and" -> language == Language.CLEANC ? " && " : " & ";
		case "or" -> language == Language.CLEANC ? " || " : " | ";
		case "==" -> language == Language.CLEANC ? " == " : " = ";
		/*
		 * Return '->' for both CleanC and LTLSMV, even though CleanC does not
		 * actually support implications. serializeBinaryExpression() will then
		 * take care of working around this lack of support using custom logic.
		 * FIXME perhaps a clearer and more elegant solution should be devised.
		 */
		case "implies" -> language == Language.LTL ? " implies " : " -> ";
		default -> " " + operatorString + " ";
		};
	}

}
