package eu.fbk.eclipse.explodtwin.ossimporter.tests;

import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Provider;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.ossimporter.core.OssModelImporter;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.sysmlv2.SysMLv2StandaloneSetup;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;

public class OssImporterTest {

	private static final String PROJECT_NAME = "TestProject";
	private static final String SYSML_EXTENSION = ".sysml";
	private static final String ORACLES_PATH = "oracles";

	private static final String FOLDABLE_DRILL = "resources/FoldableDrill/FoldableDrill.oss";
	private static final String SSR = "resources/SSR/SSR.oss";

	private static final Map<String, String> MODEL_TO_ORACLE = Map.of(
            FOLDABLE_DRILL, getDefaultOraclesLocationString(FOLDABLE_DRILL),
            SSR, getDefaultOraclesLocationString(SSR)
            );

	private static final String COMPAT_LIBRARY = "../../libs/compat/";
	private static final String FBK_LIBRARY = "../../libs/fbk.library/";

	@Inject
	private Provider<ResourceSet> resourceSetProvider;

	@Test
	public void testImportFoldableDrill() throws Exception {
		testImport(FOLDABLE_DRILL);
	}
	
	@Disabled("This test is disabled because in the export back to OSS, additional parenthesis are added in contract constraints.")
	@Test
	public void testImportSSR() throws Exception {
		testImport(SSR);
	}
	
	private void testImport(String model) throws Exception {
		createProject();
		File imported = importer(model);
		exporter(imported, model);
	}
	
	private File importer(String inputFilePath) throws Exception {
		System.out.println("\nImporter test started");
		
		File inputFile = new File(inputFilePath);
		File generated = (new OssModelImporter()).processModel(inputFile, null);
		assertTrue(generated != null, "The OSS import failed!");

		System.out.println("Generated file = " + generated);
		
		Path oraclePath = Paths.get(MODEL_TO_ORACLE.get(inputFilePath) + getOraclesFileNamePrefix(inputFilePath) + SYSML_EXTENSION);
		assertTrue(compareTwoFilesIgnoreEOL(generated.toPath(), oraclePath), "Generated SysML file is different from the oracle!");

		System.out.println("Importer test successful, generated file is equal to oracle");

		return generated;
	}		
	
	private void exporter(File inputFile, String oraclePath) throws IOException {
		File generated = generateOSS(inputFile, 1);		
		assertTrue(compareTwoFilesIgnoreEOL(generated.toPath(), Paths.get(oraclePath)), "Generated OSS file is different from the oracle!");

		System.out.println("Exporter test successful, generated file is equal to initial file");
	}
	
	/**
	 * This is necessary for the test to run, it will hold the generated files.
	 * @throws CoreException
	 */
	private void createProject() throws CoreException {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(PROJECT_NAME);
		if (project.exists()) {
			project.delete(true, new NullProgressMonitor());
		}
		project.create(new NullProgressMonitor());
		project.open(new NullProgressMonitor());
	}

	private static String getDefaultOraclesLocationString(String inputFilePath) {
		int index = inputFilePath.lastIndexOf("/");
		String userModelFolder = index == -1 ? "." : inputFilePath.substring(0, index);
		return userModelFolder + "/" + ORACLES_PATH + "/";
	}

	/**
	 *
	 * @param inputFile
	 * @param timeSpecification // 0 for hybrid model, 1 for discrete model, 2 for timed model
	 * @return 
	 * @throws IOException
	 */
	private File generateOSS(File inputFile, int timeSpecification) throws IOException {
		String inputFilePath = inputFile.getAbsolutePath().replace("\\", "/");
		PartUsage systemComponent = (PartUsage) loadModel(inputFilePath);

		String generatedOSS = AdapterUtil.generateOCRA(systemComponent, timeSpecification);

		assertTrue(generatedOSS != null, "The OCRA export failed!");
		
		return new File(generatedOSS);
	}

	private static String getOraclesFileNamePrefix(String inputFilePath) {
		int index = inputFilePath.lastIndexOf("/");
		String userInput = index == -1 ? inputFilePath : inputFilePath.substring(index + 1, inputFilePath.length());
		userInput = userInput.substring(0, userInput.lastIndexOf("."));
		return userInput;
	}

	/**
	 * Returns the top-level component of the model.
	 * @param fileName
	 * @return
	 */
	private EObject loadModel(String fileName) {
		Injector injector = new SysMLv2StandaloneSetup().createInjectorAndDoEMFRegistration();
		OssImporterTest instance = injector.getInstance(OssImporterTest.class);

		ResourceSet resourceSet = instance.resourceSetProvider.get();
		loadSysMLv2Libraries(resourceSet);

		int index = fileName.lastIndexOf("/");
		String userModelFolder = index == -1 ? "." : fileName.substring(0, index);

		findSysMLFiles(userModelFolder).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		EcoreUtil2.resolveAll(resourceSet);


		Resource resource = resourceSet.getResources().stream()
			.filter(res -> res.getURI().toString().endsWith(fileName))
			.findAny().orElseThrow();

		System.out.println("Validating model");
		for (Resource res : resourceSet.getResources()) {
			if (!res.getErrors().isEmpty()) {
				res.getErrors().forEach(System.err::println);
				throw new RuntimeException("Invalid model, check the standard error for more details.");
			}
		}

		EObject top = resource.getContents().get(0);
		return ModelUtil.getSystemPart(EcoreUtil2.getResourceSet(top));
	}

	private void loadSysMLv2Libraries(ResourceSet resourceSet) {
		findSysMLFiles(COMPAT_LIBRARY).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		findSysMLFiles(FBK_LIBRARY).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		EcoreUtil2.resolveAll(resourceSet);
	}

	private List<Path> findSysMLFiles(String directory) {
		BiPredicate<Path, BasicFileAttributes> matcher = (path, attributes) -> attributes.isRegularFile() && path.getFileName().toString().endsWith(".sysml");
	    try (var pathsStream = Files.find(Paths.get(directory), Integer.MAX_VALUE, matcher)) {
			return pathsStream.toList();
		} catch (IOException e) {
			return List.of();
		}
	}

	private boolean compareTwoFilesIgnoreEOL(Path p1, Path p2) throws IOException {
		BufferedReader reader1 = new BufferedReader(new FileReader(p1.toFile()));
		BufferedReader reader2 = new BufferedReader(new FileReader(p2.toFile()));
		String line1 = reader1.readLine();
		String line2 = reader2.readLine();
		boolean areEqual = true;

		while (line1 != null || line2 != null) {
			if (!equalsIgnoreNewlineStyle(line1, line2)) {
				areEqual = false;
				break;
			}
			line1 = reader1.readLine();
			line2 = reader2.readLine();
		}
		reader1.close();
		reader2.close();

		return areEqual;
	}

	/**
	 * Compares two strings ignoring line terminator differences.
	 *
	 * @param s1
	 * @param s2
	 * @return true if the two strings are equal
	 */
	private boolean equalsIgnoreNewlineStyle(String s1, String s2) {
	    return s1 != null && s2 != null && normalizeLineEnds(s1).equals(normalizeLineEnds(s2));
	}

	private String normalizeLineEnds(String s) {
	    return s.replace("\r\n", "\n").replace('\r', '\n').replace("&#xD;","");
	}

}
