/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ossimporter.core;

import java.util.LinkedHashSet;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.emf.common.util.BasicEMap;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EObject;

import eu.fbk.eclipse.explodtwin.ossimporter.utils.OssModelUtil;
import eu.fbk.eclipse.standardtools.utils.core.visitors.ITreeGenericVisitorAction;
import eu.fbk.tools.editor.basetype.baseType.EnumType;
import eu.fbk.tools.editor.basetype.baseType.Identifier;

/**
 * An OSS visitor to populate enumerators maps.
 *
 */
public class OssEnumVisitorAction implements ITreeGenericVisitorAction {
	private static final String ENUM_PREFIX = "Enum_";
	private int enumeratorCounter;

	/**
	 * This structure links an enumerator name to the list of values it contains.
	 */
	private EMap<String, Set<String>> enumeratorsCache = new BasicEMap<>();

	/**
	 * This structure links an enum type port to its enum definition.
	 */
	private EMap<String, String> enumPortsCache = new BasicEMap<>();


	private void action(EnumType o) throws Exception {
		addEnumerator(o);
	}

	/**
	 * Creates a new enumerator type, if needed. Updates the enumerators cache and also adds an entry in the 
	 * ports cache, linking a qualified port name to its enumerator.
	 * @param enumType
	 */
	private void addEnumerator(EnumType enumType) {
	
		// Get the enum values from the type
		Set<String> enumValuesNames = new LinkedHashSet<>(); // Keep insertion order
		for (EObject object : enumType.getValues()) {
			enumValuesNames.add(((Identifier) object).getValue());
		}

		String enumName = null;
		for (Entry<String, Set<String>> entry : enumeratorsCache) {
			if (entry.getValue().equals(enumValuesNames)) {
				enumName = entry.getKey();
			}
		}
		
		// Create a new enum type if not already present
		if (enumName == null) {			
			enumName = ENUM_PREFIX + ++enumeratorCounter;
			enumeratorsCache.put(enumName, enumValuesNames);
		}
		
		// Store the port and the enumerator in the map, if the enum is related to a port
		String componentName = OssModelUtil.getOwnerComponentName(enumType);
		String port = OssModelUtil.getPortNameForEnum(enumType);
		if (port != null) {
			enumPortsCache.put(componentName + "." + port, enumName);
		}
	}
	
	@Override
	public void genericAction(EObject o) throws Exception {
		if (o instanceof EnumType) {
			action((EnumType) o);
		}
	}

	public EMap<String, Set<String>> getEnumeratorsCache() {
		return enumeratorsCache;
	}

	public EMap<String, String> getEnumPortsCache() {
		return enumPortsCache;
	}
}
