/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ossimporter.core;

import java.io.File;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import eu.fbk.eclipse.standardtools.ModelTranslatorToOcra.core.services.OSSModelFactory;
import eu.fbk.eclipse.standardtools.utils.core.utils.FileSystemUtil;
import eu.fbk.eclipse.standardtools.utils.core.visitors.TreeVisitor;
import eu.fbk.eclipse.standardtools.utils.core.visitors.TreeVisitor.Order;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DirectoryUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.OCRADialogUtil;
import eu.fbk.tools.editor.oss.oss.OSS;

/**
 * This class converts an OSS model to a SysML v2 model.
 * 
 */
public class OssModelImporter {
	private static final Logger logger = Logger.getLogger(OssModelImporter.class);
	private static final String SYSML_SUFFIX = "sysml";
	public OCRADialogUtil ocraDialogUtil = OCRADialogUtil.getInstance();
	public OSSModelFactory ossModelFactory = OSSModelFactory.getInstance();
	public DialogUtil dialogUtil = DialogUtil.getInstance();
	public DirectoryUtil directoryUtil = DirectoryUtil.getInstance();

	/**
	 * Given a OSS model file, returns a SysML v2 file.
	 * @param ossFile the input file
	 * @param targetPath the destination folder
	 * @return the output file
	 * @throws Exception
	 */
	public File processModel(File ossFile, String targetPath) throws Exception {
		final OSS ossModel = ossModelFactory.createOssModel(ossFile);

		// Check the name of the root component of the model, and fix it if necessary
		if (!Character.isUpperCase(ossModel.getSystem().getType().charAt(0))) {
			logger.warn("The root component is not written in uppercase, fixing it");
			ossModel.getSystem().setType(StringUtils.capitalize(ossModel.getSystem().getType()));
		}

		// Visit the OSS model to extract the enumerators
		final OssEnumVisitorAction enumVisitorAction = new OssEnumVisitorAction();
		final TreeVisitor enumVisitor = new TreeVisitor(enumVisitorAction, Order.POSTORDER);
		enumVisitor.run(ossModel);
		
		// Visit the OSS model to create the SysML model
		final OssVisitorAction visitorAction = new OssVisitorAction(enumVisitorAction);
		final TreeVisitor visitor = new TreeVisitor(visitorAction, Order.POSTORDER);
		visitor.run(ossModel);
		
		// Retrieve the generated model
		String outputText = (String) visitorAction.getOutputObject(ossModel);
		String modelName = ossModel.getSystem().getType();
		
		final File sysMLFile;
		if (targetPath != null) {
			sysMLFile = new File(targetPath + File.separator + modelName + "." + SYSML_SUFFIX);
		} else {

			// Store the model in the active project folder
			sysMLFile = new File(directoryUtil.getCurrentProjectDir() + File.separator + modelName + "." + SYSML_SUFFIX);		
		}
		FileSystemUtil.writeFile(sysMLFile, outputText);

		logger.debug("PATH = " + sysMLFile.getAbsolutePath());
		logger.debug("MODEL = \n" + outputText);

		return sysMLFile;
	}
}