/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ossimporter.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import java.util.StringJoiner;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.BasicEMap;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.EcoreUtil2;

import com.google.common.primitives.Ints;

import eu.fbk.eclipse.explodtwin.ossimporter.utils.OssModelUtil;
import eu.fbk.eclipse.standardtools.ModelTranslatorToOcra.core.utils.OSSModelUtil;
import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.eclipse.standardtools.utils.core.visitors.ICachedTreeVisitorSupport;
import eu.fbk.tools.editor.basetype.baseType.*;
import eu.fbk.tools.editor.contract.contract.*;
import eu.fbk.tools.editor.contract.expression.expression.*;
import eu.fbk.tools.editor.oss.validation.ModelUtil;
import eu.fbk.tools.editor.oss.oss.*;

/**
 * An Ecore visitor to generate SysMLv2 model from an OSS model.
 *
 */
public class OssVisitorAction implements IOssVisitorAction, ICachedTreeVisitorSupport {
	private static final Logger logger = Logger.getLogger(OssVisitorAction.class);
	protected static final String RESERVED_SUFFIX = "_v";
	protected static final String RANGE_PREFIX = "Bounded";
	protected static final String IFF = " iff ";
	protected static final String NOT = " not ";
	protected static final String IMPLIES = " implies ";
	protected OSSModelUtil ossModelUtil = OSSModelUtil.getInstance();
	protected boolean importContracts;
	protected boolean importLogicFunctions;
	protected boolean importScalarValues;
	protected boolean importMathFunctions;
	protected boolean importExtraTypes;
	protected boolean importTimeModes;
	protected boolean importLTLSpec;
	protected boolean importLTLFunction;
	protected boolean importExploDTwinMetadata;
	protected boolean importExploDTwinTypes;
	
	/**
	 * This is the set of SysMLv2 and KERML reserved words that should not be used as names of elements.
	 */
	protected Set<String> reservedKeywords = new HashSet<>(Arrays.asList("about", "abstract", "accept", "action", "actor",
			"after", "alias", "all", "allocate", "allocation", "analysis", "and", "as", "assert", "assign", "assoc", "assume",
			"at", "attribute", "behavior", "bind", "binding", "block", "bool", "by", "calc", "case", "chains", "class",
			"classifier", "comment", "composite", "concern", "conjugate", "conjugates", "conjugation", "connect", "connection",
			"connector", "constraint", "datatype", "decide", "def", "default", "defined", "dependency", "derived",
			"differences", "disjoining", "disjoint", "do", "doc", "element", "else", "end", "entry", "enum", "event",
			"exhibit", "exit", "expose", "expr", "false", "feature", "featured", "featuring", "filter", "first", "flow",
			"for", "fork", "frame", "from", "function", "hastype", "if", "implies", "import", "in", "include", "individual",
			"inout", "interaction", "interface", "intersects", "inv", "inverse", "inverting", "istype", "item", "join",
			"language", "loop", "member", "merge", "message", "metaclass", "metadata", "multiplicity", "namespace",
			"nonunique", "not", "null", "objective", "occurrence", "of", "or", "ordered", "out", "package", "parallel",
			"part", "perform", "port", "portion", "predicate", "private", "protected", "public", "readonly", "redefines",
			"redefinition", "ref", "references", "relationship", "render", "rendering", "rep", "require", "requirement",
			"return", "satisfy", "send", "snapshot", "specialization", "specializes", "stakeholder", "state", "step",
			"struct", "subclassifier", "subject", "subset", "subsets", "subtype", "succession", "then", "timeslice",
			"to", "transition", "true", "type", "typed", "typing", "unions", "until", "use", "variant", "variation",
			"verification", "verify", "via", "view", "viewpoint", "when", "while", "xor"));

	/**
	 * This structure links an enumerator name to the list of values it contains. It is computed by another visitor.
	 */
	protected EMap<String, Set<String>> enumeratorsCache;

	/**
	 * This structure links an enum type port to its enum definition. It is computed by another visitor.
	 */
	protected EMap<String, String> enumPortsCache;

	/**
	 * This structure links a component to the list of its ports.
	 */
	protected EMap<String, Set<String>> componentPortsCache = new BasicEMap<>();

	/**
	 * This structure links a port to its direction and type
	 */
	protected EMap<String, Pair<String, String>> portsCache = new BasicEMap<>();
	
	/**
	 * This structure links a rangetype to its boundaries.
	 */
	protected EMap<String, Pair<Number, Number>> rangeTypeCache = new BasicEMap<>();
	
	/**
	 * This structure links a range port to its range type.
	 */
	protected EMap<String, String> rangePortsCache = new BasicEMap<>();

	/**
	 * This structure contains all the interpreted functions
	 */
	protected EMap<String, InterpretedFunctionBody> interpretedFunctions = new BasicEMap<>();
	protected record InterpretedFunctionBody(String body, List<InterpretedFunctionParameter> parameters) {}
	protected record InterpretedFunctionParameter(String name, String type) {}

	public OssVisitorAction() {
		
		// If not cleared, the cache could contain elements from a previous interaction
		translatorCache.clear();
	}
	
	public OssVisitorAction(OssEnumVisitorAction enumVisitorAction) {
		this();
		enumeratorsCache = enumVisitorAction.getEnumeratorsCache();
		enumPortsCache = enumVisitorAction.getEnumPortsCache();
	}

	@Override
	public void action(Assumption o) throws Exception {
		putOutputObject(o, getOutputObject(o.getConstraint()));
	}

	@Override
	public void action(Guarantee o) throws Exception {
		putOutputObject(o, getOutputObject(o.getConstraint()));
	}

	@Override
	public void action(TemporalExpression o) throws Exception {

		// Bounds need to be handled, but they are not present in the LTL library yet
		putOutputObject(o, getOutputObject(o.getOp()) + " ( { " + removeParenthesis(o.getLeft()) + " }, { " + removeParenthesis(o.getRight()) + " } ) ");
	}

	@Override
	public void action(TimeAnnotation o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(RealType o) throws Exception {
		putOutputObject(o, "Real");
		importScalarValues = true;
	}

	@Override
	public void action(RealLiteral o) throws Exception {
		putOutputObject(o, o.getValue());
	}

	@Override
	public void action(IntegerType o) throws Exception {
		putOutputObject(o, "Integer");
		importScalarValues = true;
	}

	@Override
	public void action(ContinuousType o) throws Exception {
		putOutputObject(o, "Continuous");
		importContracts = true;
	}

	@Override
	public void action(ComplexType o) throws Exception {
		putOutputObject(o, getOutputObject(o.getComplexType()));
	}

	@Override
	public void action(InputPort o) throws Exception {
		addPortToCache(o);
		portsCache.put((String) getOutputObject(o.getId()), new Pair<String, String>("in", (String) getOutputObject(o.getType())));
	}
	
	@Override
	public void action(OutputPort o) throws Exception {
		addPortToCache(o);
		portsCache.put((String) getOutputObject(o.getId()), new Pair<String, String>("out", (String) getOutputObject(o.getType())));
	}
	
	/**
	 * Adds the given port name to the list of ports of its owner component.
	 * @param port
	 * @throws Exception
	 */
	private void addPortToCache(Port port) throws Exception {
		String componentName = OssModelUtil.getOwnerComponentName(port);
		Set<String> ports = componentPortsCache.get(componentName);
		if (ports == null) {
			ports = new HashSet<>();
		}
		ports.add((String) getOutputObject(port.getId()));
		componentPortsCache.put(componentName, ports);
	}
	
	@Override
	public void action(InterfaceInstance o) throws Exception {
		if (o.getVariable() != null) {
			// Skip variables for now, they will be handled later when also INVAR have been processed, inside handleInterface
		} else if (o.getContract() != null) {
			// Skip contracts for now, they will be handled later when also refinements have been processed, inside handleInterface
		} else if (o.getDefine() != null) {
			// Skip defines for now, they will be handled later when creating the global structure, inside handleInterface
		} else if (o.getAssertion() != null) {
			putOutputObject(o, getOutputObject(o.getAssertion()));
		} else {
			throw new Exception("ParameterAssumption element not handled");
		}
	}

	@Override
	public void action(BooleanType o) throws Exception {
		putOutputObject(o, "Boolean");
		importScalarValues = true;
	}

	@Override
	public void action(EventType o) throws Exception {
		putOutputObject(o, "event");
		importExploDTwinTypes = true;
	}

	@Override
	public void action(PortId o) throws Exception {
		
		// Change the port name in the OSS node if it is a reserved keyword
		if (reservedKeywords.contains(o.getName())) {
			o.setName(o.getName() + RESERVED_SUFFIX);
			logger.warn("A reserved key is used, adding " + RESERVED_SUFFIX + " suffix.");
		}
		putOutputObject(o, o.getName());
	}

	@Override
	public void action(Identifier o) throws Exception {
		putOutputObject(o, o.getValue());
	}

	@Override
	public void action(BooleanLiteral o) throws Exception {
		putOutputObject(o, o.getValue().toLowerCase());
	}

	@Override
	public void action(EnumType o) throws Exception {
		String componentName = OssModelUtil.getOwnerComponentName(o);
		String ownerPort = OssModelUtil.getPortNameForEnum(o);
		String enumName = enumPortsCache.get(componentName + "." + ownerPort);
		putOutputObject(o, enumName);		
	}

	@Override
	public void action(FullPortId o) throws Exception {
		putOutputObject(o, ossModelUtil.getFullPortIdToString(o, false));
	}

	@Override
	public void action(EqualityOperator o) throws Exception {
		if (o.getName().equals("=")) {
			putOutputObject(o, " == ");
		} else {
			putOutputObject(o, " != ");
		}
	}

	@Override
	public void action(IntegerLiteral o) throws Exception {
		putOutputObject(o, o.getValue());
	}

	@Override
	public void action(EqualityExpression o) throws Exception {
		
		// Special handling when one operator is an enum value
		String rightEnum = getEnum(o.getRight());
		if (rightEnum != null) {
			FullPortId port = getFullPortIdFromExpression(o.getLeft());
			String enumDef = getPortEnumeration(port);
			putOutputObject(o.getRight(), enumDef + "::" + rightEnum);
		}		
		
		String leftEnum = getEnum(o.getLeft());
		if (leftEnum != null) {
			FullPortId port = getFullPortIdFromExpression(o.getLeft());
			String enumDef = getPortEnumeration(port);
			putOutputObject(o.getLeft(), enumDef + "::" + leftEnum);
		}

		putOutputObject(o, (String) getOutputObject(o.getLeft()) + getOutputObject(o.getOp())  + getOutputObject(o.getRight()));
	}

	/**
	 * Given a port, return its enumeration type, if any.
	 * @param port
	 * @return
	 * @throws Exception
	 */
	private String getPortEnumeration(FullPortId port) throws Exception {
		String portId = getPortOfFullPortId(port);
		String owner = getPortOwnerTypeName(port);
		String enumDef = enumPortsCache.get(owner + "." + portId);
		return enumDef;
	}

	/**
	 * Return the name of type of the owner of the port.
	 * @param port
	 * @return
	 * @throws Exception
	 */
	private String getPortOwnerTypeName(FullPortId port) throws Exception {
		String owner = getComponentsOfFullPortId(port);
		if (owner == null) {
			
			// Port of the same component, retrieve my owner
			owner = OssModelUtil.getOwnerComponentName(port);
		} else {
		
			// Port of another component, determine the type of the subcomp
			owner = OssModelUtil.getSubComponentType(port, owner);
		}
		return owner;
	}

	/**
	 * Returns the first FullPortId contained in the given expression.
	 * @param expr
	 * @return
	 */
	private FullPortId getFullPortIdFromExpression(Expression expr) {
		if (expr instanceof FullPortId) {
			return (FullPortId) expr;
		} else {
			return EcoreUtil2.getAllContentsOfType(expr, FullPortId.class).get(0);
		}
	}
	
	/**
	 * Checks if the given expression is an enum value and returns it, if any.
	 * @param expr
	 * @return
	 * @throws Exception
	 */
	private String getEnum(Expression expr) throws Exception {
		if (OssModelUtil.isSimplePortId(expr)) {
			FullPortId port = (FullPortId) expr;
			String portId = getPortOfFullPortId(port);
			String owner = OssModelUtil.getOwnerComponentName(port);
			if (!OssModelUtil.isValue(portId) && !isRegularPort(owner, portId)) {
				return portId;
			}
		}
		return null;
	}
	
	/**
	 * Returns true if the given port is a port of the owner.
	 * @param owner
	 * @param port
	 * @return
	 */
	private boolean isRegularPort(String owner, String port) {
		Set<String> ports = componentPortsCache.get(owner);
		return (ports != null) ? ports.contains(port) : false;
	}
	
	/**
	 * Returns the component part of a given FullPortId.
	 * @param id
	 * @return
	 * @throws Exception
	 */
	private String getComponentsOfFullPortId(FullPortId id) throws Exception {
		if ((id.getFullComponentIds() != null) && (!id.getFullComponentIds().isEmpty())) {
			StringJoiner joiner = new StringJoiner(".");
			for (ComponentId comp : id.getFullComponentIds()) {
				joiner.add((String) getOutputObject(comp));
			}
			return joiner.toString();
		}
		return null;
	}
	
	/**
	 * Returns the port part of a given FullPortId.
	 * @param id
	 * @return
	 * @throws Exception
	 */
	private String getPortOfFullPortId(FullPortId id) throws Exception {
		return (String) getOutputObject(id.getId());
	}
	
	@Override
	public void action(BoundedExpression o) throws Exception {
		putOutputObject(o, " ( " + getOutputObject(o.getExpression()) + " ) ");
	}

	@Override
	public void action(LogicalOrOperator o) throws Exception {
		putOutputObject(o, " or ");
	}

	@Override
	public void action(LogicalAndOperator o) throws Exception {
		putOutputObject(o, " and ");
	}

	@Override
	public void action(LogicalExpression o) throws Exception {

		// Handle the special case of the iff operator
		if (getOutputObject(o.getOp()).equals(IFF)) {
			putOutputObject(o, IFF + "( { " + removeParenthesis(o.getLeft()) + " }, { " + removeParenthesis(o.getRight()) + " } ) ");
		} else {
			putOutputObject(o, (String) getOutputObject(o.getLeft()) + getOutputObject(o.getOp())  + getOutputObject(o.getRight()));
		}
	}

	@Override
	public void action(Operator o) throws Exception {

		// Operators not handled by their specific nodes are handled here
		String operator = switch (o.getName()) {
			case "in the future" -> "in_the_future";
			case "in the past" -> "in_the_past";
			case "at next" -> "at_next";
			case "at last" -> "at_last";
			default -> o.getName();
		};
		
		putOutputObject(o, " " + operator + " ");
	}

	@Override
	public void action(NextFunction o) throws Exception {
		putOutputObject(o, o.getName() + " ( { " + removeParenthesis(o.getArgument()) + " } ) ");
	}

	@Override
	public void action(ExtendedLogicalOperator o) throws Exception {
		String operatorName = o.getName();
		if (operatorName.equals("iff") || operatorName.equals("<->")) {
			putOutputObject(o, IFF);
			importLogicFunctions = true;
		} else if (operatorName.equals("implies") || operatorName.equals("->")) {
			putOutputObject(o, IMPLIES);
		}
	}

	@Override
	public void action(UnaryTemporalExpression o) throws Exception {

		// Bounds need to be handled, but they are not present in the LTL library yet
		
		putOutputObject(o, getOutputObject(o.getOperator()) + " ( { " + removeParenthesis(o.getOperand()) + " } ) ");
	}

	@Override
	public void action(UnaryLogicalOperator o) throws Exception {
		putOutputObject(o, NOT);
	}

	@Override
	public void action(Contract o) throws Exception {
		importContracts = true;
		importLTLFunction = true;

		// No need to handle this, it is handled in a upper level
	}

	@Override
	public void action(ChangeFunction o) throws Exception {
		putOutputObject(o, o.getName() + " ( { " + removeParenthesis(o.getArgument()) + " } ) ");
		
	}

	@Override
	public void action(FallRiseFunction o) throws Exception {
		putOutputObject(o, o.getName() + " ( { " + removeParenthesis(o.getArgument()) + " } ) ");
	}

	@Override
	public void action(TimeSinceUntilFunction o) throws Exception {
		putOutputObject(o, o.getName() + " ( { " + removeParenthesis(o.getArgument()) + " } ) ");
	}

	@Override
	public void action(UnaryLogicalExpression o) throws Exception {
		putOutputObject(o, (String) getOutputObject(o.getOperator()) + getOutputObject(o.getOperand()));
	}

	/**
	 * Removes the parenthesis from a BoundedExpression.
	 * @param expr
	 * @return
	 * @throws Exception
	 */
	private String removeParenthesis(Expression expr) throws Exception {
		if (expr instanceof BoundedExpression) {
			return ((String) getOutputObject(expr)).replaceAll("^\\s*\\(\\s*|\\s*\\)\\s*$", "");
		} else {
			return (String) getOutputObject(expr);
		}
	}
	
	@Override
	public void action(Interface o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(SystemComponent o) throws Exception {
		String systemComponentName = o.getType() == null ? "System" : o.getType();
		putOutputObject(o, createBody(o, systemComponentName));
	}
	
	@Override
	public void action(OSS o) throws Exception {
		SystemComponent system = o.getSystem();
		String systemName = system.getType();
		
		String top = "package " + systemName + " {";
		top += System.lineSeparator() + "\tprivate import Hierarchy::*;";
		
		if (importContracts) {
			top += System.lineSeparator() + "\tprivate import Contracts::*;";
		}
		if (importScalarValues) {
			top += System.lineSeparator() + "\tprivate import ScalarValues::*;";
		}
		if (importLogicFunctions) {
			top += System.lineSeparator() + "\tprivate import LogicFunctions::*;";
		}
		
		if (importMathFunctions) {
			top += System.lineSeparator() + "\tprivate import MathFunctions::*;";
		}
		
		if (importExtraTypes) {
			top += System.lineSeparator() + "\tprivate import ExtraTypes::*;";
		}

		if (importTimeModes) {
			top += System.lineSeparator() + "\tprivate import TimeModes::*;";
		}

		if (importLTLSpec) {
			top += System.lineSeparator() + "\tprivate import LTLSpec::*;";
		}

		if (importLTLFunction) {
			top += System.lineSeparator() + "\tprivate import LTLFunctions::*;";
		}

		if (importExploDTwinMetadata) {
			top += System.lineSeparator() + "\tprivate import ExploDTwinMetadata::*;";
		}

		top += System.lineSeparator();

		if (importExploDTwinTypes) {
			top += System.lineSeparator() + "\tabstract item def Event;";
			top += System.lineSeparator() + "\titem def event :> Event;";
			top += System.lineSeparator();
		}

		// Create the enumerators section
		for (Entry<String, Set<String>> enumerator : enumeratorsCache) {
			String enumDef = System.lineSeparator() + "\tenum def " + enumerator.getKey() + " {";
			for (String value : enumerator.getValue()) {
				enumDef += System.lineSeparator() + "\t\tenum " + value + ";";
			}
			enumDef += System.lineSeparator() + "\t}";

			top += enumDef + System.lineSeparator();
		}
		
		// Create the range type section
		for (Entry<String, Pair<Number, Number>> rangeType : rangeTypeCache) {
			String rangeDef = System.lineSeparator() + "\tattribute def " + rangeType.getKey();
			if (rangeType.getValue().getLeft() instanceof Integer) {
				rangeDef += " :> BoundedInt {";
			} else {
				rangeDef += " :> BoundedReal {";
			}
			rangeDef += System.lineSeparator() + "\t\tattribute :>> minValue = " + rangeType.getValue().getLeft() + ";";
			rangeDef += System.lineSeparator() + "\t\tattribute :>> maxValue = " + rangeType.getValue().getRight() + ";";
			rangeDef += System.lineSeparator() + "\t}";
			top += rangeDef + System.lineSeparator();
		}
		
		// Create the interpreted functions section
		for (String function : interpretedFunctions.keySet()) {
			String define = System.lineSeparator() + "\tcalc def " + function + " {";
			InterpretedFunctionBody body = interpretedFunctions.get(function);
			for (InterpretedFunctionParameter param : body.parameters) {
				define += System.lineSeparator() + "\t\t" + param.name + " : " + param.type + ";";
			}
			define += System.lineSeparator() + "\t\t" + body.body;
			define += System.lineSeparator() + "\t}";
			top += define + System.lineSeparator();
		}
		
		// Create the part def for the system component
		top += getOutputObject(system);
		
		// Create the part defs for the other components
		for (Component component : o.getComponents()) {
			top += getOutputObject(component);
		}
		
		// Create the root part usage
		top += System.lineSeparator() + "\tpart " + StringUtils.uncapitalize(systemName) + 
				" : " + systemName + ", SystemComponent {";
		
		top += System.lineSeparator() + "\t}";		
		top += System.lineSeparator() + "}";
		
		putOutputObject(o,  top);
	}

	@Override
	public void action(ComponentId o) throws Exception {
		putOutputObject(o, o.getName());
	}

	@Override
	public void action(SubComponentType o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(SubComponent o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(RefinementInstance o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(IterativeCondition o) throws Exception {
		if (o.getConstraint() != null || o.getIteratorBounds() != null) {
			throw new Exception("IterativeCondition in connections is not supported yet");
		}
	}

	/**
	 * INVARs could be a range constraint for ports, but also something else.
	 * If the format is like "portName >= 0 and portName <= 20" it will be used to create a range type.
	 */
	@Override
	public void action(InvarBehaviour o) throws Exception {
		
		// Check to see if only a single port is mentioned in the expression
		List<FullPortId> ports = EcoreUtil2.getAllContentsOfType(o, FullPortId.class);
		if (ports.size() == 2 && ports.get(0).getId().getName().equals(ports.get(1).getId().getName())) {
			String portName = ports.get(0).getId().getName();

			// Check to see if the expression is an and of the relational operators
			if (o.getConstraint() instanceof LogicalExpression logicalExpression
					&& logicalExpression.getOp() instanceof LogicalAndOperator) {
				if (logicalExpression.getLeft() instanceof RelationalExpression left && left.getOp() instanceof RelationalOperator
						&& logicalExpression.getRight() instanceof RelationalExpression right && right.getOp() instanceof RelationalOperator) {
					
					// Extract the bounds, they could be reals or integers
					List<EObject> bounds = new ArrayList<>();
					bounds.addAll(EcoreUtil2.getAllContentsOfType(logicalExpression, IntegerLiteral.class));
					bounds.addAll(EcoreUtil2.getAllContentsOfType(logicalExpression, RealLiteral.class));
					if (bounds.size() == 2) {

						// Create the range type and associate the port to that type
						String rangeName = createRangeType(bounds.get(0), bounds.get(1));
						rangePortsCache.put(portName, rangeName);
					} else {
						throw new Exception("Cannot handle this type of bounded element");
					}
				}
			}
		} else {
			throw new Exception("Cannot handle this type of InvarBehaviour");
		}
	}

	@Override
	public void action(Connection o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(AddSubExpression o) throws Exception {
		putOutputObject(o, (String) getOutputObject(o.getLeft()) + getOutputObject(o.getOp())  + getOutputObject(o.getRight()));
	}

	@Override
	public void action(RelationalExpression o) throws Exception {
		putOutputObject(o, (String) getOutputObject(o.getLeft()) + getOutputObject(o.getOp())  + getOutputObject(o.getRight()));		
	}

	@Override
	public void action(UnaryAlgebraicExpression o) throws Exception {
		putOutputObject(o, (String) getOutputObject(o.getOperator()) + getOutputObject(o.getOperand()));
	}

	@Override
	public void action(Component o) throws Exception {
		String componentName = o.getType();				
		putOutputObject(o, createBody(o, componentName));
	}

	private String createBody(AbstractComponent o, String componentName) throws Exception {
		String body = System.lineSeparator() + "\tpart def " + componentName + " {";

		// Add ports, contracts, etc.
		body += handleInterface(o.getInterface());
		body += handleRefinement(o.getRefinement());
		body += System.lineSeparator() + "\t}" + System.lineSeparator();

		// Clear the caches for the next iteration
		portsCache.clear();
		rangePortsCache.clear();
		return body;
	}

	private String handleInterface(Interface interfaceElement) throws Exception {
		String body = "";
		
		// Handle all the in/out attributes
		for (Entry<String, Pair<String, String>> port : portsCache) {
			String portName = port.getKey();
			String direction = port.getValue().getLeft();
			String portType = rangePortsCache.get(portName);
			if (portType == null) {
				portType = port.getValue().getRight();
			}
			body += System.lineSeparator() + "\t\t" + direction + " attribute " + portName + " : " + portType + ";";
		}
		body += System.lineSeparator();
		
		// Handle the remaining elements
		for (InterfaceInstance interfaceInstance : interfaceElement.getInterfaces()) {
			if (interfaceInstance.getContract() != null) {
				Contract ctr = interfaceInstance.getContract();
				String contract = "attribute " + ctr.getName() + " : Contract {";
				if (ctr.getAssumption() != null) {
					contract += System.lineSeparator() + "\t\t\tassert constraint :>> assumption {";
					contract += System.lineSeparator() + "\t\t\t\t" + 
							ossModelUtil.getCleanExpression((String) getOutputObject(ctr.getAssumption()));
					contract += System.lineSeparator() + "\t\t\t}";
				}
				if (ctr.getGuarantee() != null) {
					contract += System.lineSeparator() + "\t\t\tassert constraint :>> guarantee {";
					contract += System.lineSeparator() + "\t\t\t\t" +
							ossModelUtil.getCleanExpression((String) getOutputObject(ctr.getGuarantee()));
					contract += System.lineSeparator() + "\t\t\t}";
				}
				String refinement = findContractRefinement(ctr);
				if (refinement != null) {
					contract += System.lineSeparator() + "\t\t\t" + refinement;
				}
				contract += System.lineSeparator() + "\t\t}";
				body += System.lineSeparator() + "\t\t" + contract;
			} else if (interfaceInstance.getDefine() != null) {

				// Process the Define and store it the global list, will be added later to the model
				handleDefine(interfaceInstance.getDefine());
			} else if (interfaceInstance.getVariable() != null) {
				// Skip variables, they are already handled at the top
			} else {
				body += System.lineSeparator() + "\t\t" + getOutputObject(interfaceInstance);
			}
		}
		return body;
	}

	/**
	 * Processes the Define, creating an entry in the interpretedFunctions list if necessary.
	 * @param define
	 * @throws Exception
	 */
	private void handleDefine(Define define) throws Exception {
		String functionName = define.getName();
		if (!interpretedFunctions.containsKey(functionName)) {
			String body = (String) getOutputObject(define.getConstraint());
			List<InterpretedFunctionParameter> parameters = new ArrayList<>();
			parameters = computeParameters(define);			
			InterpretedFunctionBody functionBody = new InterpretedFunctionBody(body, parameters);
			interpretedFunctions.put(functionName, functionBody);
		}
	}
	
	/**
	 * Returns the list of the parameters necessary for the definition of the Define.
	 * @param define
	 * @return
	 * @throws Exception
	 */
	private List<InterpretedFunctionParameter> computeParameters(Define define) throws Exception {
		List<InterpretedFunctionParameter> parameters = new ArrayList<>();
		
		// Get the interface that contains ports and defines
		Interface iface = ModelUtil.getParentComponentType(define).getInterface();
		
		// Loop on all the ports found in the Define
		for (FullPortId port : EcoreUtil2.getAllContentsOfType(define, FullPortId.class)) {
			String portName = getPortOfFullPortId(port);
			for (InterfaceInstance instance : iface.getInterfaces()) {
				if (instance.getVariable() != null) {
					Variable var = instance.getVariable();
					if (var.getId().getName().equals(portName)) {
						InterpretedFunctionParameter parameter = 
								new InterpretedFunctionParameter(portName, (String) getOutputObject(((Port) var).getType().getComplexType()));
						parameters.add(parameter);
					}
				}
			}
		}		
		return parameters;
	}

	/**
	 * Given a contract, look into the refinements to see if it is refined, and return the refinement.
	 * @param contract
	 * @return
	 * @throws Exception
	 */
	private String findContractRefinement(Contract contract) throws Exception {
		AbstractComponent owner = ModelUtil.getParentComponentType(contract);
		Refinement refinement = owner.getRefinement();
		if (refinement != null) {
			for (RefinementInstance ref : refinement.getRefinements()) {
				if (ref.getRefinedby() != null) {
					RefinedBy refined = ref.getRefinedby();
					if (refined.getName().equals(contract.getName())) {
						return (String) getOutputObject(refined);
					}
				}
			}
		}
		return null;
	}

	/**
	 * Method to create the composition of elements based on the refinement.
	 * @param refinement the refinement to analyze
	 * @return
	 * @throws Exception
	 */
	private String handleRefinement(Refinement refinement) throws Exception {
		String composition = "";
		if (refinement != null) {
			for (RefinementInstance refinementInstance : refinement.getRefinements()) {
				if (refinementInstance.getSubcomponent() != null) {
					SubComponent subComp = refinementInstance.getSubcomponent();
					String subCompName = subComp.getName().getName();
					String subCompType = subComp.getType().getComponentTypeName();
					composition += System.lineSeparator() + "\t\tpart " + subCompName + " : " + subCompType + ";";
				} else if (refinementInstance.getConnection() != null) {
					Connection connection = refinementInstance.getConnection();
					FullVariableId var = connection.getVariable();
					String variable = (String) getOutputObject(var);
					Expression expr = connection.getConstraint();
					String expression = (String) getOutputObject(expr);
					if (expr instanceof FullPortId) {
						if (var instanceof FullPortId) {

							// Check if the expression is an enum literal and if the variable is an enum
							String enumDef = getPortEnumeration((FullPortId) var);
							String enumLiteral = getEnum(expr);
							if (enumDef != null && enumLiteral != null) {

								// Assignment of a literal value
								composition += System.lineSeparator() + "\t\t:>> " + variable + " = " + enumDef + "::" + expression + ";";
							} else {

								// Regular composition, if at least one of the port belongs to the component, a bind is needed
								if (getComponentsOfFullPortId((FullPortId) var) == null || getComponentsOfFullPortId((FullPortId) expr) == null) {
									composition += System.lineSeparator() + "\t\tbind " + expression + " = " + variable + ";";	
								} else {
									composition += System.lineSeparator() + "\t\tconnect " + expression + " to " + variable + ";";									
								}
							}
						} else {

							// Variable could also be a FullVariable or a FullParameter. Maybe they can be handled in the same way as FullPortId
							// but this will require to refactor also other methods
							throw new Exception("Not supported variable type: " + var);
						}
					} else {
						composition += System.lineSeparator() + "\t\t:>> " + variable + " = " + expression + ";";
						importLTLSpec = true;	// the expression could contain LTL operators, enable the relative import
						importLTLFunction = true;
					}
				} else if (refinementInstance.getRefinedby() != null) {
					// Do not handle Refinedby element, it is already used in interface contract definition
				} else if (refinementInstance.getAssertion() != null) {
					Assertion assertion = refinementInstance.getAssertion();
					composition += getOutputObject(assertion);
				} else if (refinementInstance.getBehaviour() != null) {
					if (!(refinementInstance.getBehaviour() instanceof InvarBehaviour)) {
						throw new Exception("Refinement element not handled: " + refinementInstance);
					}
				} else {
					
					// Other types of refinement are not handled yet: FormulaConstraint | ValidProp
					throw new Exception("Refinement element not handled: " + refinementInstance);
				}
			}
		}
		return composition;
	}

	@Override
	public void action(Refinement o) throws Exception {

		// No need to handle this
	}

	@Override
	public void action(RangeType o) throws Exception {
		if (o.getLowerBound() instanceof Identifier || o.getUpperBound() instanceof Identifier) {
			throw new Exception("Identifier element not supported in RangeType");
		} else {
			String rangeName = createRangeType(o.getLowerBound(), o.getUpperBound());
			putOutputObject(o, rangeName);
		}
	}

	/**
	 * Creates a new range type in the cache, if needed. Bounds can be IntegerLiteral or RealLiteral.
	 * @param bound1 first bound, not necessarily the lower bound
	 * @param bound2 second bound, not necessarily the upper bound
	 * @return the name of the range type
	 */
	private String createRangeType(EObject bound1, EObject bound2) throws Exception {
		String rangeName = null;

		importExtraTypes = true;

		// Extract the values of the bounds, as strings
		String value1 = extractLiteralValue(bound1);
		String value2 = extractLiteralValue(bound2);
		
        Number lowerBound = null;
        Number upperBound = null;

		// Attempt to parse both strings as integers.
        Integer i1 = Ints.tryParse(value1);
        Integer i2 = Ints.tryParse(value2);

        if (i1 != null && i2 != null) {
            if (i1 < i2) {
            	lowerBound = i1;
            	upperBound = i2;
            } else {
            	lowerBound = i2;
            	upperBound = i1;
            }
        } else {
            try {
            	float f1 = Float.parseFloat(value1);
            	float f2 = Float.parseFloat(value2);
            	if (f1 < f2) {
            		lowerBound = f1;
            		upperBound = f2;
            	} else {
            		lowerBound = f2;
            		upperBound = f1;
            	}
            } catch (NumberFormatException e) {
                throw new Exception("Error: One of the ranges is not a valid number.");
            }
        }
		Pair<Number, Number> newPair = new Pair<>(lowerBound, upperBound);
		
		// Store the range if not already present
		for (Entry<String, Pair<Number, Number>> entry : rangeTypeCache) {
			if (entry.getValue().equals(newPair)) {
				rangeName = entry.getKey();
			}
		}
		if (rangeName == null) {
			rangeName = createRangeName(lowerBound, upperBound);
			rangeTypeCache.put(rangeName, newPair);
		}
		return rangeName;
	}

	/**
     * Creates a range name from two sorted numbers based on a template.
     * It handles negative signs and decimal points to ensure a valid identifier.
     *
     * @param lower The lower bound number (Integer or Float)
     * @param upper The upper bound number (Integer or Float)
     * @return A formatted, variable-safe string
     */
    public static String createRangeName(Number lower, Number upper) {
        String lowerStr = formatNumberForVariable(lower);
        String upperStr = formatNumberForVariable(upper);

        return String.format("%s__%s__%s", RANGE_PREFIX, lowerStr, upperStr);
    }

    /**
     * Formats a single Number into a string safe for use in a variable name.
     * Replaces '-' with 'neg' and '.' with '_'.
     */
    private static String formatNumberForVariable(Number num) {
        String s = String.valueOf(num);
        
        // Handle negative sign first
        if (s.startsWith("-")) {
            s = "neg" + s.substring(1);
        }
        
        // Handle decimal point
        return s.replace('.', '_');
    }
	
	/**
	 * Given an IntegerLiteral or RealLiteral, return the contained value.
	 * @param bound
	 * @return
	 */
	private String extractLiteralValue(EObject bound) {
		if (bound instanceof IntegerLiteral b) {
			return b.getValue();
		} else if (bound instanceof RealLiteral b) {
			return b.getValue();
		}
		return null;
	}

	@Override
	public void action(FullContractIdList o) throws Exception {
		putOutputObject(o, ossModelUtil.getFullContractIdAsString(o, false));
	}

	@Override
	public void action(Define o) throws Exception {
		
		// No need to handle this, it is handled in a upper level
	}
	
	@Override
	public void action(Assertion o) throws Exception {
		String assertion = "attribute " + ossModelUtil.getAssertionName(o) + " : LTLSpec {" + System.lineSeparator();
		assertion += "\t\t\tconstraint :>> formula = { "; 
		assertion += getOutputObject(o.getConstraint()) + "};";
		assertion += System.lineSeparator() + "\t\t}";
		putOutputObject(o, assertion);
		importLTLSpec = true;
		importLTLFunction = true;
	}
	
	@Override
	public void action(DerivativeFunction o) throws Exception {
		putOutputObject(o, o.getName() + " ( " + getOutputObject(o.getArgument()) + " ) ");
	}

	@Override
	public void action(RefinedBy o) throws Exception {
		String refinement =  ":>> refinedBy = ";
		StringJoiner joiner = new StringJoiner(", ", "(", ");");
		for (FullContractIdList idList : o.getFullContractIds()) {
			joiner.add((String) getOutputObject(idList));
		}
		refinement += joiner.toString();//
		putOutputObject(o, refinement);
	}

	@Override
	public void action(CountFunction o) throws Exception {
		importMathFunctions = true;
		String count = " " + o.getName() + " ";
		StringJoiner joiner = new StringJoiner(", ", "( ", " ) "); 
		for (Expression arg : o.getArguments()) {
			joiner.add((String) getOutputObject(arg));
		}
		count += joiner.toString();
		putOutputObject(o, count);
	}

	@Override
	public void action(ClockType o) throws Exception {
		putOutputObject(o, "Clock");
		importTimeModes = true;
	}

	@Override
	public void action(Parameter o) throws Exception {
		String parameter = null;
		if (o.getParameters().size() != 0) {
			parameter = System.lineSeparator() + "\t\tcalc def " + getOutputObject(o.getId()) + " {" + System.lineSeparator();
			parameter += "\t\t\t@UninterpretedFunction;" + System.lineSeparator();
			int counter = 0;
			for (ComplexType param : o.getParameters()) {
				parameter += "\t\t\tin attribute in" + counter++ + ": " + getOutputObject(param) + ";" + System.lineSeparator();
			}
			parameter += "\t\t\treturn : " + getOutputObject(o.getType()) + ";" + System.lineSeparator() + "\t\t}";
		} else {
			parameter = "attribute " + getOutputObject(o.getId()) + " : " + getOutputObject(o.getType()) + ";";
		}
		importExploDTwinMetadata = true;
		putOutputObject(o, parameter);
	}

	@Override
	public void action(ParameterId o) throws Exception {
		putOutputObject(o, ossModelUtil.getVariableIdAsString(o, false));
	}

	@Override
	public void action(Bound o) throws Exception {
		throw new Exception("Bounds on LTL operators are not supported yet");
	}
		
}
