package eu.fbk.eclipse.explodtwin.tests;

import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;

import org.apache.commons.lang3.SystemUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.formatting2.regionaccess.internal.TextRegionAccessBuildingSequencer;
import org.junit.jupiter.api.Test;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Provider;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.api.util.PostProcessor;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.sysmlv2.SysMLv2StandaloneSetup;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.ui.preferences.PreferenceConstants;

class ExploDTwinTest {

	private static final String OSS_EXTENSION = ".oss";
	private static final String SMV_EXTENSION = ".smv";
	private static final String ORACLES_PATH = "oracles";
	private static final String PROJECT_NAME = "TestProject";

	private static final String WINDOWS_OCRA_EXECUTABLE_PATH = "tools/ocra-2.1.0/ocra_win64.exe";
	private static final String LINUX_OCRA_EXECUTABLE_PATH = "tools/ocra-2.1.0/ocra-linux64";

	public static final String PLUGIN_ID = ExploDTwinTest.class.getPackage().getName();

	private static final String FOLDABLE_DRILL = "resources/FoldableDrill/FoldableDrill.sysml";
	private static final String SSR = "resources/SSR/SSR.sysml";
	private static final String EXOMARS = "../../../../CaseStudies/ExoMars/model/ExomarsSystemModel.sysml";

	private static final Map<String, String> MODEL_TO_ORACLE = Map.of(
            FOLDABLE_DRILL, getDefaultOraclesLocationString(FOLDABLE_DRILL),
            SSR, getDefaultOraclesLocationString(SSR),
            EXOMARS, "resources/Exomars/oracles/"
            );

	private static final String COMPAT_LIBRARY = "../../libs/compat/";
	private static final String FBK_LIBRARY = "../../libs/fbk.library/";
	private static final String EXPLODTWIN_LIBRARY = "../../libs/explodtwin.library/";

	@Inject
	private Provider<ResourceSet> resourceSetProvider;

	@Test
	public void testSSR_OSS() throws Exception {
		generateOSS(SSR, 1, false);
	}

	@Test
	public void testSSR_SMV() throws Exception {
		generateSMV(SSR, 1, false);
	}

	@Test
	public void testFoldableDrill_OSS() throws Exception {
		generateOSS(FOLDABLE_DRILL, 1, false);
	}

	@Test
	public void testFoldableDrill_SMV() throws Exception {
		generateSMV(FOLDABLE_DRILL, 1, false);
	}

	@Test
	public void testExomars_OSS() throws Exception {
		generateOSS(EXOMARS, 2, true);
	}

	@Test
	public void testExomars_SMV() throws Exception {
		generateSMV(EXOMARS, 2, true);
	}

	/**
	 *
	 * @param inputFile
	 * @param timeSpecification // 0 for hybrid model, 1 for discrete model, 2 for timed model
	 * @throws IOException
	 */
	private void generateOSS(String inputFile, int timeSpecification, boolean isExplodtwin) throws IOException {
		PartUsage systemComponent = (PartUsage) loadModel(inputFile, isExplodtwin);
		if (isExplodtwin) {
			new PostProcessor(EcoreUtil2.getResourceSet(systemComponent), isExplodtwin).performPostProcessing();
		}

		String generatedOSS = AdapterUtil.generateOCRA(systemComponent, timeSpecification);

		assertTrue(generatedOSS != null, "The OCRA export failed!");

		final Path oraclePath = Paths.get(MODEL_TO_ORACLE.get(inputFile) +
				getOraclesFileNamePrefix(inputFile) + OSS_EXTENSION);
		assertTrue(compareTwoFilesIgnoreEOL(Paths.get(generatedOSS), oraclePath),
				"Generated OSS file is different from the oracle!");
	}

	/**
	 *
	 * @param inputFile
	 * @param timeSpecification // 0 for hybrid model, 1 for discrete model, 2 for timed model
	 * @throws Exception
	 */
	@SuppressWarnings("restriction")
	private void generateSMV(String inputFile, int timeSpecification, boolean isExplodtwin) throws Exception {
		createProject();
		setOcraPath();

		/*
		 * A slightly hacky way to disable a noisy Xtext error (it really should
		 * be a warning) that clutters the CI logs and fills the whole log buffer.
		 */
		final Logger xtextLogger = Logger.getLogger(TextRegionAccessBuildingSequencer.class);
		final Level defaultLevel = xtextLogger.getLevel();
		xtextLogger.setLevel(Level.OFF);

		PartUsage systemComponent = (PartUsage) loadModel(inputFile, isExplodtwin);
		if (isExplodtwin) {
			new PostProcessor(EcoreUtil2.getResourceSet(systemComponent), isExplodtwin).performPostProcessing();
		}

		String generatedSMV = AdapterUtil.generateSMV(systemComponent, timeSpecification);

		assertTrue(generatedSMV != null, "The SMV export failed!");

		final Path oraclePath = Paths.get(MODEL_TO_ORACLE.get(inputFile) +
				getOraclesFileNamePrefix(inputFile) + SMV_EXTENSION);
		assertTrue(compareTwoFilesIgnoreEOL(Paths.get(generatedSMV), oraclePath),
				"Generated SMV file is different from the oracle!");

		xtextLogger.setLevel(defaultLevel);
	}

	private void createProject() throws CoreException {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(PROJECT_NAME);
		if (!project.exists()) {
			project.create(new NullProgressMonitor());
		}
		project.open(new NullProgressMonitor());
	}

	private static String getDefaultOraclesLocationString(String inputFilePath) {
		final int index = inputFilePath.lastIndexOf("/");
		final String userModelFolder = index == -1 ? "." : inputFilePath.substring(0, index);
		return userModelFolder + "/" + ORACLES_PATH + "/";
	}

	private static String getOraclesFileNamePrefix(String inputFilePath) {
		final int index = inputFilePath.lastIndexOf("/");
		String userInput = index == -1 ? inputFilePath : inputFilePath.substring(index + 1, inputFilePath.length());
		userInput = userInput.substring(0, userInput.lastIndexOf("."));
		return userInput;
	}

	/**
	 * Returns the top-level component of the model.
	 * @param fileName
	 * @return
	 */
	private EObject loadModel(String fileName, boolean isExplodtwin) {
		Injector injector = new SysMLv2StandaloneSetup().createInjectorAndDoEMFRegistration();
		ExploDTwinTest instance = injector.getInstance(ExploDTwinTest.class);

		final ResourceSet resourceSet = instance.resourceSetProvider.get();
		loadSysMLv2Libraries(resourceSet);
		if (isExplodtwin) {
			loadExplodtwinLibrary(resourceSet);
		}

		final int index = fileName.lastIndexOf("/");
		final String userModelFolder = index == -1 ? "." : fileName.substring(0, index);

		findSysMLFiles(userModelFolder).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		EcoreUtil2.resolveAll(resourceSet);


		Resource resource = resourceSet.getResources().stream()
			.filter(res -> res.getURI().toString().endsWith(fileName))
			.findAny().orElseThrow();

		System.out.println("Validating model");
		for (final Resource res : resourceSet.getResources()) {
			if (!res.getErrors().isEmpty()) {
				res.getErrors().forEach(System.err::println);
				throw new RuntimeException("Invalid model, check the standard error for more details.");
			}
		}

		EObject top = resource.getContents().get(0);
		return ModelUtil.getSystemPart(EcoreUtil2.getResourceSet(top));
	}

	private void loadSysMLv2Libraries(ResourceSet resourceSet) {
		findSysMLFiles(COMPAT_LIBRARY).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		findSysMLFiles(FBK_LIBRARY).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		EcoreUtil2.resolveAll(resourceSet);
	}

	private void loadExplodtwinLibrary(ResourceSet resourceSet) {
		findSysMLFiles(EXPLODTWIN_LIBRARY).stream()
			.forEach(path -> resourceSet.getResource(URI.createFileURI(path.toAbsolutePath().toString()), true));
		EcoreUtil2.resolveAll(resourceSet);
	}

	private List<Path> findSysMLFiles(String directory) {
		final BiPredicate<Path, BasicFileAttributes> matcher = (path, attributes) -> attributes.isRegularFile() && path.getFileName().toString().endsWith(".sysml");
	    try (final var pathsStream = Files.find(Paths.get(directory), Integer.MAX_VALUE, matcher)) {
			return pathsStream.toList();
		} catch (IOException e) {
			return List.of();
		}
	}

	private boolean compareTwoFilesIgnoreEOL(Path p1, Path p2) throws IOException {
		BufferedReader reader1 = new BufferedReader(new FileReader(p1.toFile()));
		BufferedReader reader2 = new BufferedReader(new FileReader(p2.toFile()));
		String line1 = reader1.readLine();
		String line2 = reader2.readLine();
		boolean areEqual = true;

		while (line1 != null || line2 != null) {
			if (!equalsIgnoreNewlineStyle(line1, line2)) {
				areEqual = false;
				break;
			}
			line1 = reader1.readLine();
			line2 = reader2.readLine();
		}
		reader1.close();
		reader2.close();

		return areEqual;
	}

	/**
	 * Compares two strings ignoring line terminator differences.
	 *
	 * @param s1
	 * @param s2
	 * @return true if the two strings are equal
	 */
	private boolean equalsIgnoreNewlineStyle(String s1, String s2) {
	    return s1 != null && s2 != null && normalizeLineEnds(s1).equals(normalizeLineEnds(s2));
	}

	private String normalizeLineEnds(String s) {
	    return s.replace("\r\n", "\n").replace('\r', '\n').replace("&#xD;","");
	}

	/**
	 * Hardcode the path of the OCRA tool, taking it from the plugin.
	 */
	private void setOcraPath() {
        IPreferenceStore store = eu.fbk.tools.adapter.ui.Activator.getDefault().getPreferenceStore();
		URL entry = Platform.getBundle(PLUGIN_ID).getEntry("/");
		try {
			String basedir = org.eclipse.core.runtime.FileLocator.toFileURL(entry).getPath();
			System.out.println("basedir = " + basedir);

        	if (SystemUtils.IS_OS_WINDOWS) {
				store.setValue(PreferenceConstants.OCRA_EXECUTABLE, basedir + WINDOWS_OCRA_EXECUTABLE_PATH);

            	System.out.println("\n\n\n FRONTEND OCRA VALUE = " + store.getString(PreferenceConstants.OCRA_EXECUTABLE));

			} else if (SystemUtils.IS_OS_LINUX) {
				store.setValue(PreferenceConstants.OCRA_EXECUTABLE, basedir + LINUX_OCRA_EXECUTABLE_PATH);
            	System.out.println("\n\n\n FRONTEND OCRA VALUE = " + store.getString(PreferenceConstants.OCRA_EXECUTABLE));
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
