/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import java.io.IOException;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.ui.editor.XtextEditor;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.utils.EditorUtils;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.api.util.PostProcessor;
import eu.fbk.eclipse.standardtools.utils.ui.utils.CommandBuilder;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.ui.commands.AbstractBaseCommand;

/**
 * The base command for the checks.
 *
 */
public abstract class AbstractBaseCheck extends AbstractBaseCommand {

	public AbstractBaseCheck() {
		super("");
	}

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		preprocessEvent(event);

		IWorkbenchPart part = HandlerUtil.getActiveWorkbenchWindow(event).getActivePage().getActivePart();
		if (part instanceof XtextEditor) {
			IXtextDocument xtextDocument = EditorUtils.getActiveXtextEditor().getDocument();
			xtextDocument.readOnly(xtextResource -> {
					processSysMLModel(xtextResource);
					return true;
				});
		}

		return null;
	}

    /**
     * Opens the V & V Results view.
     */
    protected void openVResults() {
        Display display = Display.getDefault();
        if (display != null) {
            display.asyncExec(() -> {
                    try {
                        IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
                        page.showView("eu.fbk.tools.adapter.ui.views.request.RequestView");
                    } catch (PartInitException e) {
                        e.printStackTrace();
                    }
                });
        }
    }

	/**
	 * Shows the result of the given function, reading data from the given file.
	 *
	 * @param functionName
	 *            the name of the function used to obtain the result
	 * @param fileName
	 *            the file containing the data to visualize
	 */
	public void showResult(String functionName, String fileName) {
		String commandId = "eu.fbk.tools.adapter.ui.commands.ShowVVResultCommand";
		String functionNameParam = "function_name";
		String fileNameParam = "result_file";

		CommandBuilder computeFaultTree;
		try {
			computeFaultTree = CommandBuilder.build(commandId);
			computeFaultTree.setParameter(functionNameParam, functionName);
			computeFaultTree.setParameter(fileNameParam, fileName);
			computeFaultTree.execute();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	protected PartUsage getSystemPart(ResourceSet resourceSet) {
		try {
			return ModelUtil.getSystemPart(resourceSet);
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			return null;
		}
	}

	protected ResourceSet createWorkingCopy(ResourceSet resourceSet) {
		final ResourceSet newResourceSet = new XtextResourceSet();
		resourceSet.getResources().forEach(originalResource -> {
			final EObject rootEObject = originalResource.getContents().get(0);
			final String originalResourceText = NodeModelUtils.getNode(rootEObject).getText();
			final XtextResource newResource = (XtextResource) newResourceSet.createResource(originalResource.getURI());
			try {
				newResource.reparse(originalResourceText);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
		});
		final boolean isExplodtwin = ModelUtil.isExplodtwin(newResourceSet);
		new PostProcessor(newResourceSet, isExplodtwin).performPostProcessing();
		return newResourceSet;
	}

	abstract protected void processSysMLModel(Resource resource);
}
