/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import java.util.HashMap;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.eclipse.explodtwin.api.core.SysMLv2StateMachineModel;
import eu.fbk.eclipse.explodtwin.api.core.SysMLv2SystemModel;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.standardtools.ExecOcraCommands.ui.services.OCRAExecService;
import eu.fbk.eclipse.standardtools.StateMachineTranslatorToSmv.ui.services.SmvExportServiceUI;
import eu.fbk.eclipse.standardtools.nuXmvService.ui.utils.NuXmvDirectoryUtil;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.OCRADirectoryUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.ui.Activator;
import eu.fbk.tools.adapter.ui.preferences.PreferenceConstants;

/**
 * A Command to run a check on composite contract implementation.
 *
 */
public class CheckCompositeImplementation extends AbstractBaseCheck {
	private static final Logger logger = Logger.getLogger(CheckCompositeImplementation.class);
	private SysMLv2SystemModel systemModel = new SysMLv2SystemModel();
	private final OCRAExecService ocraService = OCRAExecService.getInstance(systemModel);
	private final OCRADirectoryUtil ocraDirectoryUtil = OCRADirectoryUtil.getInstance();
	private final SmvExportServiceUI smvExportService = SmvExportServiceUI.getInstance(systemModel, new SysMLv2StateMachineModel());
	private final NuXmvDirectoryUtil nuXmvDirectoryUtil = NuXmvDirectoryUtil.getInstance();

	@Override
	protected void processSysMLModel(Resource resource) {
		if (resource.getContents().isEmpty()) {
			logger.error("Resource hasn't got a root element.");
			return;
		}

		EObject model = resource.getContents().get(0);

		// Ask to the user to select the part usage
		ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(model));
		List<PartUsage> partUsages = ModelUtil.getAllContentsOfTypeFromModel(workingCopy, PartUsage.class);
		partUsages.sort((part1, part2) -> part1.getName().compareTo(part2.getName()));

		PartUsage system = getSystemPart(workingCopy);
		if (system == null) {
			return;
		} else {
			partUsages.add(0, system);
		}

		PartUsage selectedPartUsage = AdapterUtil.getSelectedPart(activeShell, partUsages);
		if (selectedPartUsage == null) {
			return;
		}

        final int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return;
		}
        final boolean showPopups = false;
        final boolean usexTextValidation = true;
        final boolean isAsyncCommunication = true;

        try {
            final String ossDirPath = ocraDirectoryUtil.getOSSDirPath();
            final String resultFile = ocraDirectoryUtil.getCommandCheckImplementationResultPath(selectedPartUsage.getName());
    		final String workspaceDir =
    				Activator.getDefault().getPreferenceStore().getString(PreferenceConstants.TOOL_WORKSPACE);
    		final String smvMapDirPath = nuXmvDirectoryUtil.getSmvFileDirectory();
            final SubMonitor monitorSub =
            		SubMonitor.convert(new NullProgressMonitor(), "Executing compositional verification ...", 100);
            HashMap<String, String> smvPathComponentNameMap = smvExportService.exportSmv(selectedPartUsage, showPopups,
            		workspaceDir, monitorSub, timeSpecification);

            ocraService.executeCheckCompositeContractImplementation(selectedPartUsage, resource,
            		smvPathComponentNameMap, timeSpecification, isAsyncCommunication, usexTextValidation, showPopups,
            		ossDirPath, smvMapDirPath, resultFile, monitorSub, true);
        } catch (Exception e) {
        	DialogUtil.getInstance().showMessage_ExceptionError(e);
        	e.printStackTrace();
        	return;
        }

        openVResults();
	}
}
