/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import java.io.File;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.ui.editor.XtextEditor;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.utils.EditorUtils;

import eu.fbk.eclipse.explodtwin.api.core.SysMLv2StateMachineModel;
import eu.fbk.eclipse.explodtwin.api.core.SysMLv2SystemModel;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.standardtools.ModelTranslatorToOcra.ui.services.OSSTranslatorServiceUI;
import eu.fbk.eclipse.standardtools.StateMachineTranslatorToSmv.ui.services.SmvExportServiceUI;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.FileUtils;
import eu.fbk.tools.adapter.ToolFunction;
import eu.fbk.tools.adapter.ocra.CheckContractImplementation;
import eu.fbk.tools.adapter.ocra.OcraFunction.TimeModel;
import eu.fbk.tools.adapter.ui.Activator;
import eu.fbk.tools.adapter.ui.preferences.PreferenceConstants;

/**
 * A Command to check a contract implementation, based on EATA library.
 *
 */
public class CheckContractImplementationEATA extends AbstractBaseToolCommand {
	private static final Logger logger = Logger.getLogger(CheckContractImplementationEATA.class);
	private SysMLv2SystemModel systemModel = new SysMLv2SystemModel();
	private final SmvExportServiceUI smvExportService = SmvExportServiceUI.getInstance(systemModel, new SysMLv2StateMachineModel());
	private final OSSTranslatorServiceUI ossTranslatorService = OSSTranslatorServiceUI.getInstance(systemModel);
	private CheckContractImplementation function;
	private String behaviourFileName = null;
	private String ossModelFileName = null;
	private boolean processResult = false;

	public CheckContractImplementationEATA() {
		super("OCRA", "Check Contract Implementation", PreferenceConstants.OCRA_EXECUTABLE,
				PreferenceConstants.OCRA_EXECUTION_TIMEOUT);
	}

	@Override
	protected boolean initialize() {
		behaviourFileName = null;
		ossModelFileName = null;
		function = new CheckContractImplementation();

		return super.initialize();
	}

	@Override
	public boolean preprocessEvent(ExecutionEvent event) {
		final IWorkbenchPart part = HandlerUtil.getActiveWorkbenchWindow(event).getActivePage().getActivePart();
		if (part instanceof XtextEditor) {
			final IXtextDocument xtextDocument = EditorUtils.getActiveXtextEditor().getDocument();
			xtextDocument.readOnly(xtextResource -> {
					processResult = processSysMLModel(xtextResource);
					return processResult;
				});
		}
		return processResult;
	}

	private boolean processSysMLModel(Resource resource) {
		if (resource.getContents().isEmpty()) {
			logger.error("Resource hasn't got a root element.");
			return false;
		}

		EObject model = resource.getContents().get(0);

		// Ask to the user to select the part usage
		ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(model));
		List<PartUsage> partUsages = ModelUtil.getAllContentsOfTypeFromModel(workingCopy, PartUsage.class);
		partUsages.sort((part1, part2) -> part1.getName().compareTo(part2.getName()));

		PartUsage system;
		try {
			system = ModelUtil.getSystemPart(workingCopy);
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			return false;
		}
		partUsages.add(0, system);

		PartUsage selectedPartUsage = AdapterUtil.getSelectedPart(activeShell, partUsages);
		if (selectedPartUsage == null) {
			return false;
		}

        final int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return false;
		}
        final boolean showPopups = false;
        final boolean usexTextValidation = true;
        final boolean isAsyncCommunication = true;
		final String workspaceDir =
				Activator.getDefault().getPreferenceStore().getString(PreferenceConstants.TOOL_WORKSPACE);
        final SubMonitor monitorSub =
        		SubMonitor.convert(new NullProgressMonitor(), "Executing compositional verification ...", 100);

        try {
        	behaviourFileName = smvExportService.exportSingleSmv(selectedPartUsage, showPopups, workspaceDir, monitorSub, timeSpecification);

        	// Compute the OSS file for the selected component
        	final File ossFile = ossTranslatorService.exportRootComponentToOssFile(selectedPartUsage, resource,
        			timeSpecification, isAsyncCommunication, usexTextValidation, showPopups, workspaceDir, monitorSub);
        	ossModelFileName = ossFile.getAbsolutePath();
        } catch (Exception e) {
        	DialogUtil.getInstance().showMessage_ExceptionError(e);
        	e.printStackTrace();
        	return false;
        }

		// Set the function parameters
		function.setBehaviourModel(FileUtils.stringToURI(behaviourFileName));
		function.setContractModel(FileUtils.stringToURI(ossModelFileName));
		function.setOldSmvFormat(false);
		function.setComponentName(systemModel.getComponentTypeName(selectedPartUsage));

		TimeModel timeModel = TimeModel.discrete;
		switch(timeSpecification) {
		case 0:
			timeModel = TimeModel.hybrid;
			break;
		case 1:
			timeModel = TimeModel.discrete;
			break;
		case 2:
			timeModel = TimeModel.timed;
			break;
		}
		function.setTimeModel(timeModel);
		return true;
	}

	@Override
	protected ToolFunction getFunction() {
		return function;
	}

}
