/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.eclipse.explodtwin.api.core.ModelLibrary;
import eu.fbk.eclipse.explodtwin.api.core.SysMLv2SystemModel;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.ui.dialogs.CheckDiagnosabilityParametersDialog;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil.GeneratedFiles;
import eu.fbk.eclipse.standardTools.XSapExecService.services.XSapExecService;
import eu.fbk.eclipse.standardtools.nuXmvService.ui.utils.NuXmvDirectoryUtil;
import eu.fbk.eclipse.standardtools.utils.core.utils.FileSystemUtil;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.ErrorsDialogUtil;
import eu.fbk.sysmlv2.sysMLv2.AttributeUsage;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;

/**
 * A command to check if a condition is diagnosable.
 *
 */
public class CheckDiagnosability extends AbstractBaseCheck {
	private static final Logger logger = Logger.getLogger(CheckDiagnosability.class);
	private static final String OBS_NAME = "observables.obs";
	private static final String ALARM_NAME = "alarm.asl";
	private final XSapExecService xSapExecService = XSapExecService.getInstance();
	private final SysMLv2SystemModel sysML2SystemModel = new SysMLv2SystemModel();
	protected final NuXmvDirectoryUtil nuXmvDirectoryUtil = NuXmvDirectoryUtil.getInstance();
	private final ModelLibrary modelLibrary = new ModelLibrary();

	@Override
	protected void processSysMLModel(Resource resource) {
		if (resource.getContents().isEmpty()) {
			logger.error("Resource hasn't got a root element.");
			return;
		}

		EObject rootElement = resource.getContents().get(0);

		final int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return;
		}

		// Generate the SMV for the whole system
		ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(rootElement));
		final String smvPath;
		PartUsage system = getSystemPart(workingCopy);
		if (system != null) {
			smvPath = AdapterUtil.generateSMV(system, timeSpecification);
		} else {
			ErrorsDialogUtil.getInstance().showMessage_GenericError("No system components are defined in the model, please define one.");
			return;
		}

		// Compute all the required files for the analysis
		GeneratedFiles files = null;
		try {
			files = AdapterUtil.prepareExpandedFiles(system, smvPath);
			if (files == null) {
				return;
			}
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			e.printStackTrace();
		}

		// Compute the observables, including the root name
		final String rootName = ModelUtil.getPartDefinition(system).getName();
		List<String> observablesList = computeObservables(system, null);
		if (observablesList == null) {
			return;
		}

		final CheckDiagnosabilityParametersDialog dialog = new CheckDiagnosabilityParametersDialog(activeShell, observablesList);
		dialog.open();

		if (dialog.goAhead()) {
			final String engine = dialog.getAlgorithmType();	// alternatives: bdd, bmc, ic3, msat_bmc
			final String alarmPattern = dialog.getType();
			final String alarmPath = null;	// alternative to condition
			final String condition = dialog.getCondition();
			final int delayBound = dialog.getDelayBound();
			final int bmcLength = dialog.getBoundLength();
			final String context = dialog.getContext();
			final List<String> observables = addRootName(dialog.getObservables(), rootName);

			// Store the observables and the alarm in files
			File obsFile = null;
			File alarmFile = null;
			try {
				obsFile = createObservablesFile(nuXmvDirectoryUtil.getSmvTempDirectory(), OBS_NAME, observables);
				alarmFile = createAlarmFile(nuXmvDirectoryUtil.getSmvTempDirectory(), ALARM_NAME, condition,
						alarmPattern, context, delayBound);
			} catch (Exception e) {
				DialogUtil.getInstance().showMessage_ExceptionError(e);
				e.printStackTrace();
			}

			final String filesMessage =
					"Alarm file is stored in: " + alarmFile.getAbsolutePath()
					+ System.lineSeparator() + System.lineSeparator()
					+ "Observables file is stored in: " + obsFile.getAbsolutePath();

			DialogUtil.getInstance().showMessage_GenericMessage("Check Diagnosability", filesMessage);

			final String observablesPath = obsFile.getAbsolutePath();
			final boolean needMsat = ModelUtil.usesInfiniteDomainVariables(rootElement);
			final boolean internalExecution = true;

			xSapExecService.checkDiagnosability(files.extendedSmvFileName(), engine, bmcLength, alarmPattern, alarmPath, observablesPath,
					condition, delayBound, context, needMsat, internalExecution);

			// Visualize the result
			openVResults();
		}
	}

	/**
	 * Add the root name if the observable belongs to a sub component.
	 * @param observables the observables list
	 * @param rootName the root name
	 * @return
	 */
	private List<String> addRootName(List<String> observables, String rootName) {
		final List<String> qualObservables = new ArrayList<String>();

		for (String observable : observables) {
			qualObservables.add(observable.contains(".") ? rootName + "." + observable : observable);
		}

		return qualObservables;
	}

	/**
	 * Creates a file containing the alarm as specified by the user.
	 * @param smvTempDirectory
	 * @param alarmName
	 * @param condition
	 * @param alarmPattern
	 * @param context
	 * @param delayBound
	 * @return
	 * @throws IOException
	 */
	private File createAlarmFile(String smvTempDirectory, String alarmName, String condition, String alarmPattern,
			String context, int delayBound) throws IOException {
		String content = "NAME: alarm1" + System.lineSeparator();
		content = content.concat("CONDITION: " + condition + System.lineSeparator());
		if (alarmPattern != null && !alarmPattern.isBlank()) {
			content = content.concat("TYPE: " + alarmPattern + System.lineSeparator());
		}
		if (context != null && !context.isBlank()) {
			content = content.concat("CONTEXT: " + context + System.lineSeparator());
		}
		if (delayBound > 0) {
			content = content.concat("DELAY: " + delayBound + System.lineSeparator());
		}
		final File alarmFile = new File(smvTempDirectory + File.separator + alarmName);
		FileSystemUtil.writeFile(alarmFile, content);

		return alarmFile;
	}

	/**
	 * Creates the file containing the observables.
	 * @param smvTempDirectory the location where to store the file
	 * @param obsName the name of the file
	 * @param observables the list of observables
	 * @return
	 * @throws IOException
	 */
	private File createObservablesFile(String smvTempDirectory, String obsName, List<String> observables)
			throws IOException {
		String content = "";

		for (String observable : observables) {
			content = content.concat(observable + System.lineSeparator());
		}
		final File obsFile = new File(smvTempDirectory + File.separator + obsName);
		FileSystemUtil.writeFile(obsFile, content);

		return obsFile;
	}

	/**
	 * Determines the list of all the possible observables for the given element and subcomponents.
	 * @param component the root element
	 * @param prefix the prefix to add to subcomponents variables
	 * @return
	 */
	private List<String> computeObservables(PartUsage component, String prefix) {
		final List<String> observables = new ArrayList<String>();

		// Add all the observable variables of the component
		for (AttributeUsage observable : modelLibrary.getObservableVariables(component)) {
			if (ModelUtil.isInfiniteDomainVariable(observable)) {
				displayError("CheckDiagnosability", "The variable \"" + ((NamedElement) observable.eContainer()).getName() +
						":" + observable.getName() + "\" is of infinite domain type and cannot be observed.");
			return null;
			} else {
				observables.add(addToPrefix(prefix, observable.getName()));
			}
		}

		// Add the state variable if the component has a state machine
		if (ModelUtil.getNominalStateMachinesOfComponent(component).size() != 0) {
			observables.add(addToPrefix(prefix, "State"));
		}

		// Process the subcomponents
		for(Object instance : sysML2SystemModel.getSubComponentsInstances(component)) {
			final String newPrefix = addToPrefix(prefix, sysML2SystemModel.getName(instance));
			observables.addAll(computeObservables((PartUsage) instance, newPrefix));
		}

		return observables;
	}

	/**
	 * Concatenates the given istance to the existing prefix.
	 */
	private String addToPrefix(String prefix, String instance) {
		return (prefix != null) ? prefix + "." + instance : instance;
	}
}
