/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import static org.eclipse.ui.PlatformUI.getWorkbench;

import java.util.List;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.ui.editor.XtextEditor;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.utils.EditorUtils;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.ui.dialogs.ModelCheckingParametersDialog;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil.GeneratedFiles;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.FileUtils;
import eu.fbk.tools.adapter.ToolFunction;
import eu.fbk.tools.adapter.nuxmv.CheckModel;
import eu.fbk.tools.adapter.ui.preferences.PreferenceConstants;

/**
 * A Command to run the model checker, based on EATA library. The EATA dialog is used
 * because it is used a customized dialog to collect the parameters and provides more
 * flexibility. It is possible to use models extended with faults.
 *
 */
public class CheckFaultSmvModelEATA extends AbstractBaseToolCommand {
	private CheckModel function;
	private String behaviourFileName;
	private boolean useFaultyModel;
	private EObject rootElement;
	private PartUsage selectedPartUsage;
	private PartUsage system;

	public CheckFaultSmvModelEATA() {
		super("nuXmv", "", PreferenceConstants.NUXMV_EXECUTABLE, PreferenceConstants.NUXMV_EXECUTION_TIMEOUT);
	}

	@Override
	protected boolean initialize() {
		behaviourFileName = null;
		function = new CheckModel();
		return super.initialize();
	}

	@Override
	protected ToolFunction getFunction() {
		return function;
	}

	@Override
	public boolean preprocessEvent(ExecutionEvent event) {
		if (!super.preprocessEvent(event)) {
			return false;
		}

		final IWorkbenchPart part = HandlerUtil.getActiveWorkbenchWindow(event).getActivePage().getActivePart();
		final int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return false;
		}

		if (part instanceof XtextEditor) {
			final IXtextDocument xtextDocument = EditorUtils.getActiveXtextEditor().getDocument();
			behaviourFileName = xtextDocument.readOnly(xtextResource -> {
				rootElement = xtextResource.getContents().get(0);
				ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(rootElement));
				List<PartUsage> partUsages = ModelUtil.getAllContentsOfTypeFromModel(workingCopy, PartUsage.class);
				partUsages.sort((part1, part2) -> part1.getName().compareTo(part2.getName()));

				try {
					system = ModelUtil.getSystemPart(workingCopy);
				} catch (Exception e) {
					DialogUtil.getInstance().showMessage_ExceptionError(e);
					return null;
				}
				partUsages.add(0, system);
				selectedPartUsage = AdapterUtil.getSelectedPart(activeShell, partUsages);
				if (selectedPartUsage == null) {
					return null;
				}

				return AdapterUtil.generateSMV(selectedPartUsage, timeSpecification);
			});
		}

		if (behaviourFileName == null) {
			return false;
		}

		// Set the function parameters
		function.setBehaviourModel(FileUtils.stringToURI(behaviourFileName));
		if (timeSpecification == 2) {
			function.setTimed(true);
		}

		function.enableAbstraction(ModelUtil.usesInfiniteDomainVariables(rootElement));

		// Show the dialog to collect parameters
		final Display display = getWorkbench().getDisplay();
		final ModelCheckingParametersDialog dialog = new ModelCheckingParametersDialog(activeShell,
				"Model Checking Analysis Parameters", function, asyncExecution, selectedPartUsage, system);
		display.syncExec(dialog::open);
		if (dialog.goAhead()) {
			asyncExecution = dialog.getAsyncExecution();
			useFaultyModel = dialog.isFaultModel();
		} else {
			return false;
		}

		// The extension of the model can be done only if the selected part is the root
		if (selectedPartUsage == system && useFaultyModel) {
			try {
				GeneratedFiles files = AdapterUtil.prepareExpandedFiles(system, behaviourFileName);
				if (files == null) {
					return false;
				}
				behaviourFileName = files.extendedSmvFileName();
				function.setBehaviourModel(FileUtils.stringToURI(behaviourFileName));
			} catch (Exception e) {
				DialogUtil.getInstance().showMessage_ExceptionError(e);
				e.printStackTrace();
			}
		}

		// Add the context at the end of the property
		function.setFormula(function.getFormula() + " IN " + ModelUtil.getPartDefinition(selectedPartUsage).getName());

		return true;
	}

}
