/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.ui.dialogs.ComputeFailureModesParametersDialog;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil.GeneratedFiles;
import eu.fbk.eclipse.standardTools.XSapExecService.services.XSapExecService;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.ErrorsDialogUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.xsap.ComputeFmeaTable;

/**
 * A command to run the Failure Mode Effects Analysis.
 *
 */
public class FailureModeEffectsAnalysis extends AbstractBaseCheck {
	private static final Logger logger = Logger.getLogger(FailureModeEffectsAnalysis.class);
	private final XSapExecService xSapExecService = XSapExecService.getInstance();

	@Override
	protected void processSysMLModel(Resource resource) {
		if (resource.getContents().isEmpty()) {
			logger.error("Resource hasn't got a root element.");
			return;
		}

		EObject model = resource.getContents().get(0);

		final int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return;
		}

		// Generate the SMV for the whole system
		ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(model));
		final String smvPath;
		PartUsage system = getSystemPart(workingCopy);
		if (system != null) {
			smvPath = AdapterUtil.generateSMV(system, timeSpecification);
		} else {
			ErrorsDialogUtil.getInstance().showMessage_GenericError("No system components are defined in the model, please define one.");
			return;
		}

		// Compute all the required files for the analysis
		GeneratedFiles files = null;
		try {
			files = AdapterUtil.prepareExpandedFiles(system, smvPath);
			if (files == null) {
				return;
			}
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			e.printStackTrace();
		}

		final ComputeFailureModesParametersDialog dialog = new ComputeFailureModesParametersDialog(activeShell);

		dialog.open();

		if (dialog.goAhead()) {

			// Compute the analysis
			if (xSapExecService.computeFmea(files.extendedSmvFileName(), files.fmsFileName(), dialog.getProperties(),
					files.ftFileName(), true, ModelUtil.getPartDefinition(system).getName(), dialog.getAlgorithmType(),
					dialog.getBoundLength(), dialog.getCardinality())) {
			}

			// Visualize the result
			showResult(ComputeFmeaTable.FUNCTION_NAME, files.ftFileName());
//			openVResults();
		}
	}

}
