/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.standardtools.ModelTranslatorToOcra.ui.utils.OCRATranslatorDialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;

/**
 * A Command to export to OSS a single component or the whole model.
 *
 */
public class GenerateOssModel extends AbstractBaseCheck {
	private static final Logger logger = Logger.getLogger(GenerateOssModel.class);

	@Override
	protected void processSysMLModel(Resource resource) {
		if (resource.getContents().isEmpty()) {
			logger.error("Resource hasn't got a root element.");
			return;
		}

		EObject model = resource.getContents().get(0);

		ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(model));
		List<PartUsage> partUsages = ModelUtil.getAllContentsOfTypeFromModel(workingCopy, PartUsage.class);
		partUsages.sort((part1, part2) -> part1.getName().compareTo(part2.getName()));

		PartUsage system = getSystemPart(workingCopy);
		if (system == null) {
			return;
		} else {
			partUsages.add(0, system);
		}

		int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return;
		}

		PartUsage selectedPartUsage = AdapterUtil.getSelectedPart(activeShell, partUsages);
		if (selectedPartUsage == null) {
			return;
		}

		String ossFileName = AdapterUtil.generateOCRA(selectedPartUsage, timeSpecification);
		if (ossFileName != null) {
			OCRATranslatorDialogUtil.getInstance().showMessage_ExportArchitectureDone(ossFileName);
		} else {
			DialogUtil.getInstance().showMessage_GenericMessage("OSS Export error",
					"Error during export to OSS.");
		}
	}
}
