/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.commands;

import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.ui.dialogs.OptimizeObservablesParametersDialog;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil;
import eu.fbk.eclipse.explodtwin.util.AdapterUtil.GeneratedFiles;
import eu.fbk.eclipse.standardTools.XSapExecService.services.XSapExecService;
import eu.fbk.eclipse.standardtools.utils.ui.dialogs.MessageTimeModelDialog;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.ErrorsDialogUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
/**
 * A command to optimize the observables list.
 *
 */
public class OptimizeObservables extends AbstractBaseCheck {
	private static final Logger logger = Logger.getLogger(OptimizeObservables.class);
	private final XSapExecService xSapExecService = XSapExecService.getInstance();

	@Override
	protected void processSysMLModel(Resource resource) {
		if (resource.getContents().isEmpty()) {
			logger.error("Resource hasn't got a root element.");
			return;
		}

		EObject rootElement = resource.getContents().get(0);

		final int timeSpecification = MessageTimeModelDialog.openQuestion(true);
		if (timeSpecification < 0) {
			return;
		}

		// Generate the SMV for the whole system
		ResourceSet workingCopy = createWorkingCopy(EcoreUtil2.getResourceSet(rootElement));
		final String smvPath;
		PartUsage system = getSystemPart(workingCopy);
		if (system != null) {
			smvPath = AdapterUtil.generateSMV(system, timeSpecification);
		} else {
			ErrorsDialogUtil.getInstance().showMessage_GenericError("No system components are defined in the model, please define one.");
			return;
		}

		// Compute all the required files for the analysis
		GeneratedFiles files = null;
		try {
			files = AdapterUtil.prepareExpandedFiles(system, smvPath);
			if (files == null) {
				return;
			}
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			e.printStackTrace();
		}


		final OptimizeObservablesParametersDialog dialog = new OptimizeObservablesParametersDialog(activeShell);
		dialog.open();

		if (dialog.goAhead()) {
			final String engine = dialog.getAlgorithmType();	// alternatives: bmc_ic3, bmc
			final int bmcLength = dialog.getBoundLength();
			final String observablesPath = dialog.getObservablesPath();
			final String alarmPath = dialog.getAlarmPath();
			final boolean needMsat = ModelUtil.usesInfiniteDomainVariables(rootElement);
			final boolean internalExecution = true;

			xSapExecService.optimizeObservables(files.extendedSmvFileName(), engine, bmcLength, null, alarmPath, observablesPath,
					null, 0, null, needMsat, internalExecution);

			openVResults();
		}
	}
}
