/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.dialogs;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;

import eu.fbk.tools.adapter.xsap.XsapFunction.AlgorithmType;

/**
 * A class to collect the parameters to run the check diagnosability.
 *
 */
public class CheckDiagnosabilityParametersDialog extends Dialog {
	private Combo algorithmTypeCombo;
	private Combo alarmTypeCombo;
	private Text conditionText;
	private Text contextText;
	private Spinner delayBoundSpinner;
	private Spinner boundLengthSpinner;
	private List<String> observableList;
	private Table observables;
	private Button allObservables;

	private String[] algorithmTypes = { "bdd", "bmc", "ic3", "msat_bmc" };
	private String[] alarmTypes = { "finite", "exact", "bounded" };

	/** The go ahead. */
	private boolean goAhead = false;
	private String condition;
	private String context;
	private String alarmType;
	private int boundLength;
	private int delayBound;
	private AlgorithmType algorithmType;
	private List<String> selectedObservables;

	/**
	 * Create the dialog.
	 *
	 * @param parentShell
	 * @param observableList 
	 */
	public CheckDiagnosabilityParametersDialog(Shell parentShell, List<String> observableList) {
		super(parentShell);
		setShellStyle(SWT.TITLE | SWT.RESIZE);
		this.observableList = observableList;
	}

	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("Check Diagnosability Parameters");
	}

	/** {@inheritDoc} */
	@Override
	protected void okPressed() {
		condition = conditionText.getText();
		alarmType = alarmTypeCombo.getItem(alarmTypeCombo.getSelectionIndex());
		context = contextText.getText();
		delayBound = delayBoundSpinner.getSelection();
		algorithmType = AlgorithmType.valueOf(algorithmTypeCombo.getItems()[algorithmTypeCombo.getSelectionIndex()]);
		boundLength = boundLengthSpinner.getSelection();

		selectedObservables = new ArrayList<String>();
		for (TableItem item : observables.getItems()) {
			if (item.getChecked()) {
				selectedObservables.add(item.getText());
			}
		}

		goAhead = true;

		super.okPressed();
	}

	@Override
	protected void cancelPressed() {
		goAhead = false;

		super.cancelPressed();
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		parent.setToolTipText("");
		final Composite container = (Composite) super.createDialogArea(parent);

		final Composite dialogContainer = new Composite(container, SWT.NONE);
		dialogContainer.setLayout(new GridLayout(2, false));
		dialogContainer.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		final Label title1 = new Label(dialogContainer, SWT.NONE);
		final FontDescriptor boldDescriptor = FontDescriptor.createFrom(title1.getFont()).setStyle(SWT.BOLD);
		final Font boldFont = boldDescriptor.createFont(title1.getDisplay());
		title1.setFont(boldFont);
		title1.setText("Configure the alarm:");
		title1.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
		
		final Label lblProperties = new Label(dialogContainer, SWT.NONE);
		lblProperties.setText("Condition");
		conditionText = new Text(dialogContainer, SWT.SINGLE | SWT.BORDER);
		conditionText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		conditionText.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event e) {
				getButton(IDialogConstants.OK_ID).setEnabled(!conditionText.getText().equals(""));
			}
		});

		final Label lblAlarmType = new Label(dialogContainer, SWT.NONE);
		lblAlarmType.setText("Type");
		alarmTypeCombo = new Combo(dialogContainer, SWT.NONE | SWT.READ_ONLY);
		alarmTypeCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		setComboValues(alarmTypeCombo, alarmTypes);
		alarmTypeCombo.select(0);

		final Label lblContext = new Label(dialogContainer, SWT.NONE);
		lblContext.setText("Context");
		contextText = new Text(dialogContainer, SWT.SINGLE | SWT.BORDER);
		contextText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final Label lblDelayBound = new Label(dialogContainer, SWT.NONE);
		lblDelayBound.setText("Delay Bound");
		delayBoundSpinner = new Spinner(dialogContainer, SWT.BORDER);
		delayBoundSpinner.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		delayBoundSpinner.setMinimum(0);
		delayBoundSpinner.setMaximum(100);
		delayBoundSpinner.setIncrement(1);
		delayBoundSpinner.setPageIncrement(10);

		Label emptyLine1 = new Label(dialogContainer, SWT.HORIZONTAL);
		emptyLine1.setLayoutData(new GridData(GridData.FILL_HORIZONTAL, GridData.FILL_VERTICAL, true, false, 2, 1));
		
		final Label title2 = new Label(dialogContainer, SWT.NONE);
		title2.setFont(boldFont);
		title2.setText("Configure the engine:");
		title2.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
		
		final Label lblAlgorithm = new Label(dialogContainer, SWT.NONE);
		lblAlgorithm.setText("Algorithm Type");
		algorithmTypeCombo = new Combo(dialogContainer, SWT.NONE | SWT.READ_ONLY);
		algorithmTypeCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		setComboValues(algorithmTypeCombo, algorithmTypes);
		algorithmTypeCombo.select(3);

		final Label lblBoundLength = new Label(dialogContainer, SWT.NONE);
		lblBoundLength.setText("Bound Length");
		boundLengthSpinner = new Spinner(dialogContainer, SWT.BORDER);
		boundLengthSpinner.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		boundLengthSpinner.setMinimum(0);
		boundLengthSpinner.setMaximum(100);
		boundLengthSpinner.setSelection(10);
		boundLengthSpinner.setIncrement(1);
		boundLengthSpinner.setPageIncrement(10);

		Label emptyLine2 = new Label(dialogContainer, SWT.HORIZONTAL);
		emptyLine2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL, GridData.FILL_VERTICAL, true, false, 2, 1));

		final Label title3 = new Label(dialogContainer, SWT.NONE);
		title3.setFont(boldFont);
		title3.setText("Select the observable variables:");
		title3.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
		
	    allObservables = new Button(dialogContainer, SWT.CHECK);
	    allObservables.setText("Select all observables");
	    allObservables.setSelection(false);
	    allObservables.addListener(SWT.Selection, new Listener() {
	    	@Override
	    	public void handleEvent(Event event) {
	    		if (allObservables.getSelection()) {
	    			setSelected(observables, true);
	    		}
	    		else {
	    			setSelected(observables, false);
	    		}
	    	}

	    	private void setSelected(Table observables, boolean selected) {
	    		for (TableItem item : observables.getItems()) {
	    			item.setChecked(selected);
	    		}
	    	}
	    });

		final GridData tableGridData = new GridData(SWT.FILL, SWT.FILL, true, true, 2, 1);
		tableGridData.heightHint=145;
		observables = new Table(dialogContainer, SWT.CHECK | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		observables.setLayoutData(tableGridData);
		for (String observable : observableList) {
			TableItem item = new TableItem(observables, SWT.NONE);
			item.setText(observable);
		}
		
		return container;
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
		createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);

		// Start with the OK button disabled, it will be enabled only if the condition is defined
        getButton(IDialogConstants.OK_ID).setEnabled(false);
	}

	@Override
	protected Point getInitialSize() {
		return new Point(500, 560);
	}

	public boolean goAhead() {
		return goAhead;
	}

	private void setComboValues(Combo combo, String[] admittedValues) {
		combo.setItems(admittedValues);
	}

	public String getCondition() {
		return condition;
	}

	public int getBoundLength() {
		return boundLength;
	}

	public String getAlgorithmType() {
		return algorithmType.toString();
	}

	public String getType() {
		return alarmType;
	}

	public int getDelayBound() {
		return delayBound;
	}

	public String getContext() {
		return context;
	}

	public List<String> getObservables() {
		return selectedObservables;
	}
}
