/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.dialogs;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;

import eu.fbk.tools.adapter.xsap.XsapFunction.AlgorithmType;

/**
 * A class to collect the parameters to run the Fault Tree Analysis.
 *
 */
public class ComputeFaultTreeParametersDialog extends Dialog {
	private Combo algorithmTypeCombo;
	private Text propertiesText;
	private Spinner boundLengthSpinner;

	private String[] algorithmTypes = { "bdd", "bmc", "ic3", "msat" };

	/** The go ahead. */
	private boolean goAhead = false;
	private String properties;
	private int boundLength;
	private AlgorithmType algorithmType;

	/**
	 * Create the dialog.
	 *
	 * @param parentShell
	 */
	public ComputeFaultTreeParametersDialog(Shell parentShell) {
		super(parentShell);
		setShellStyle(SWT.TITLE | SWT.RESIZE);
	}

	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("FTA Parameters");
	}

	/** {@inheritDoc} */
	@Override
	protected void okPressed() {
		algorithmType = AlgorithmType.valueOf(algorithmTypeCombo.getItems()[algorithmTypeCombo.getSelectionIndex()]);
		properties = propertiesText.getText();
		boundLength = boundLengthSpinner.getSelection();

		goAhead = true;

		super.okPressed();
	}

	@Override
	protected void cancelPressed() {
		goAhead = false;

		super.cancelPressed();
	}

	/** {@inheritDoc} */
	@Override
	protected Control createDialogArea(Composite parent) {
		parent.setToolTipText("");
		final Composite container = (Composite) super.createDialogArea(parent);

		final Composite container2 = new Composite(container, SWT.NONE);
		container2.setLayout(new GridLayout(2, false));
		container2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final Label lblProperties = new Label(container2, SWT.NONE);
		lblProperties.setText("TLE Property");

		propertiesText = new Text(container2, SWT.SINGLE | SWT.BORDER);
		propertiesText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final Label lblAlgorithm = new Label(container2, SWT.NONE);
		lblAlgorithm.setText("Algorithm Type");

		algorithmTypeCombo = new Combo(container2, SWT.NONE | SWT.READ_ONLY);
		algorithmTypeCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		setComboValues(algorithmTypeCombo, algorithmTypes);

		final Label lblBoundLength = new Label(container2, SWT.NONE);
		lblBoundLength.setText("Bound Length");

		boundLengthSpinner = new Spinner(container2, SWT.BORDER);
		boundLengthSpinner.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		boundLengthSpinner.setMinimum(0);
		boundLengthSpinner.setMaximum(50);
		boundLengthSpinner.setSelection(0);
		boundLengthSpinner.setIncrement(1);
		boundLengthSpinner.setPageIncrement(5);

		return container;
	}

	/** {@inheritDoc} */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
		createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
	}

	/** {@inheritDoc} */
	@Override
	protected Point getInitialSize() {
		return new Point(450, 200);
	}

	/** Getter. */
	public boolean goAhead() {
		return goAhead;
	}

	private void setComboValues(Combo combo, String[] admittedValues) {
		combo.setItems(admittedValues);
		combo.select(3);
	}

	public String getProperties() {
		return properties;
	}

	public int getBoundLength() {
		return boundLength;
	}

	public String getAlgorithmType() {
		final String type = algorithmType.toString();

		// Special case for the "ic3" algorithm, it is the default algorithm in EATA
		if ("ic3".equals(type)) {
			return null;
		}
		
		return type;
	}

}
