/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.dialogs;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Shell;

import eu.fbk.sysmlv2.sysMLv2.NamedElement;

/**
 * A dialog to select an Element from the list.
 *
 */
public class ElementSelectorDialog extends Dialog {
	private List list;
	private String selection;
	private String text;
	private Button okButton;
	private Map<String, NamedElement> map = new LinkedHashMap<>();
	private final String title;

	/**
	 * Creates the dialog
	 * @param parentShell
	 * @param title
	 * @param text
	 */
	public ElementSelectorDialog(Shell parentShell, String title, String text) {
		super(parentShell);

		selection = null;
		this.title = title;
		this.text = text;
	}

	public String getTitle() {
		return title;
	}

	/**
	 * Create contents of the dialog.
	 *
	 * @param parent
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		getShell().setText(title);

		Composite container = (Composite) super.createDialogArea(parent);
		GridLayout gridLayout = (GridLayout) container.getLayout();
		gridLayout.marginWidth = 10;
		gridLayout.marginHeight = 5;

		Label lblTitle = new Label(container, SWT.NONE);
		lblTitle.setText(text);

		list = new List(container, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL);
		GridData gd_list = new GridData(SWT.LEFT, SWT.CENTER, true, true, 1, 1);
		gd_list.heightHint = 130;
		gd_list.widthHint = 400;
		list.setLayoutData(gd_list);

		map.values().forEach(p -> list.add(p.getName()));

		list.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				final String[] selections = list.getSelection();
				if (selections == null || selections.length != 1) {
					selection = null;
					okButton.setEnabled(false);
				} else {
					selection = selections[0];
					okButton.setEnabled(true);
				}
			}
		});

		list.addMouseListener(new MouseAdapter() {

			@Override
			public void mouseDoubleClick(MouseEvent e) {
				final String[] selections = list.getSelection();
				if (selections == null || selections.length != 1) {
					selection = null;
					okButton.setEnabled(false);
				} else {
					selection = selections[0];
					close();
				}
			}
		});

		list.pack();

		return container;
	}

	/**
	 * Create contents of the button bar.
	 *
	 * @param parent
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		okButton = createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
		okButton.setEnabled(false);
		createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
	}

	/**
	 * Return the initial size of the dialog.
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(450, 300);
	}

	public NamedElement getSelection() {
		if (selection == null) {
			return null;
		}

		return map.get(selection);
	}

	public void addElements(Collection<? extends NamedElement> elements) {
		elements.forEach(element -> map.put(element.getName(), element));
	}
}
