/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.dialogs;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.widgets.WidgetFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.eclipse.explodtwin.api.core.ModelLibrary;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.tools.adapter.nuxmv.CheckModel;

/**
 * A dialog for the model checker. It includes an additional checkbox to use extended models with faults.
 *
 */
public class ModelCheckingParametersDialog extends eu.fbk.tools.adapter.ui.dialog.behaviour.ModelCheckingParametersDialog {

	protected Button fault;
	protected boolean faultModel;
	protected Boolean enableFault;
	private final List<PartUsage> subsystemsInstances;
	private final PartUsage selectedPartUsage;
	private final ModelLibrary modelLibrary = new ModelLibrary();
	private final boolean isExplodtwin;

	public ModelCheckingParametersDialog(Shell parentShell, String title, CheckModel function, Boolean asyncExecution, PartUsage selectedPartUsage, PartUsage system) {
		super(parentShell, title, function, asyncExecution);
		this.enableFault = selectedPartUsage == system;
		this.selectedPartUsage = selectedPartUsage;
		isExplodtwin = ModelUtil.isExplodtwin(EcoreUtil2.getResourceSet(system));
		subsystemsInstances = isExplodtwin ? modelLibrary.getSubsystems(modelLibrary.getRealAsset(system)) : List.of();
	}

	// Create a composite with standard margins and spacing
	protected Control createBaseDialogArea(Composite parent) {
		GridLayout layout = new GridLayout();
		layout.marginHeight = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
		layout.marginWidth = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
		layout.verticalSpacing = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
		layout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
		Composite composite = WidgetFactory.composite(SWT.NONE).layout(layout)
				.layoutData(new GridData(GridData.FILL_BOTH)).create(parent);
		applyDialogFont(composite);
		return composite;
	}

	/** {@inheritDoc} */
	@Override
	protected Control createDialogArea(Composite parent) {
		parent.setToolTipText("");
		final Composite container = (Composite) createBaseDialogArea(parent);

	    timedDomainCheckBox = new Button(container, SWT.CHECK);
	    timedDomainCheckBox.setText("Timed Domain");
	    timedDomainCheckBox.setSelection(function.isTimed());
	    timedDomainCheckBox.setEnabled(false);

		final Composite container2 = new Composite(container, SWT.NONE);
		container2.setLayout(new GridLayout(2, false));
		GridData gd_container2 = new GridData(SWT.FILL, SWT.FILL, false, false);
		gd_container2.widthHint = 405;
		gd_container2.heightHint = 228;
		container2.setLayoutData(gd_container2);

		final Label lblCheck = new Label(container2, SWT.NONE);
		lblCheck.setText("Check Type");

		final GridData gd_comboCheckType = new GridData(SWT.LEFT, SWT.CENTER, true, false, 1, 1);
		gd_comboCheckType.widthHint = 289;

		comboCheckType = new Combo(container2, SWT.NONE | SWT.READ_ONLY);
		comboCheckType.setLayoutData(gd_comboCheckType);
		setComboValues(comboCheckType, checkTypes, function.getCheckType() == null ? "" : function.getCheckType().name());

		final Label lblAlgorithm = new Label(container2, SWT.NONE);
		lblAlgorithm.setText("Algorithm Type");

		final GridData gd_comboAlgorithmType = new GridData(SWT.LEFT, SWT.CENTER, true, false, 1, 1);
		gd_comboAlgorithmType.widthHint = 286;

		comboAlgorithmType = new Combo(container2, SWT.NONE | SWT.READ_ONLY);
		comboAlgorithmType.setLayoutData(gd_comboAlgorithmType);
		setComboValues(comboAlgorithmType, algorithmTypes, function.getAlgorithmType() == null ? algorithmTypes[2] : function.getAlgorithmType().name());

		final Label lblProperty = new Label(container2, SWT.NONE);
		lblProperty.setText("Property");

	    propertyText = new Text(container2, SWT.MULTI | SWT.BORDER | SWT.WRAP | SWT.V_SCROLL);
	    propertyText.setLayoutData(new GridData(GridData.FILL_BOTH));
	    if( !StringUtils.isBlank(function.getFormula()) )
	    {
	    	propertyText.setText(function.getFormula());
	    }

	    asyncCheckBox = new Button(container, SWT.CHECK);
	    asyncCheckBox.setText("OSLC Asynchronous Execution");
	    asyncCheckBox.setSelection(asyncExecution);
	    asyncCheckBox.setVisible(false);

	    fault = new Button(container, SWT.CHECK);
	    fault.setText("Use model extended with faults (top level component must be selected)");
	    fault.setSelection(false);
	    fault.setEnabled(enableFault);
	    fault.setToolTipText("The checkbox is enabled only if the top level component has been selected");

	    new Label(container, SWT.NONE);
	    new Label(container, SWT.NONE);
	    new Label(container, SWT.NONE);

		final Composite container3 = new Composite(container, SWT.NONE);
		GridLayout gl_container3 = new GridLayout(1, false);
		container3.setLayout(gl_container3);
		container3.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));

		return container;
	}

	/*
	 * Mathces strings such as xyz.State and returns the list of words
	 * preceding the dot ("xyz" in the example).
	 */
	public static List<String> findTokensPrefixingState(String input) {
		List<String> tokens = new ArrayList<>();
		Pattern pattern = Pattern.compile("(\\w+)\\.State");
		Matcher matcher = pattern.matcher(input);

		while (matcher.find()) {
			tokens.add(matcher.group(1));
		}

		return tokens;
	}

	/*
	 * Matches "State" when not preceded by a dot or an alphanumeric char.
	 * The goal is to detect references to "State" when it appears unqualified,
	 * because the user selected a Subsystem as the component to perform model
	 * checking on: we need to detect e.g. "State=...", "...!=State",
	 * "...=... & State=...", etc.
	 * Returns the list of indexes at which State was matched.
	 */
	public static List<Integer> findUnprefixedState(String input) {
		Pattern pattern = Pattern.compile("(?<![\\\\dA-Za-z\\\\.])State|^State");
		Matcher matcher = pattern.matcher(input);
		List<Integer> indexes = new ArrayList<>();

		while (matcher.find()) {
			indexes.add(matcher.start());
		}

		return indexes;
    }

	private String processProperty(String originalProperty) {
		String property = originalProperty;
		final String dot = ".";
		final String dotState = ".State";
		final List<String> statePrefixes =
				!ModelUtil.isSubsystem(selectedPartUsage) ? findTokensPrefixingState(property) : List.of();
		final List<Integer> unprefixedStateIndices =
				ModelUtil.isSubsystem(selectedPartUsage) ? findUnprefixedState(property) : List.of();

		/*
		 * Due to the model architecture, the two processing steps are mutually exclusive:
		 * - if the user selected a Subsystem, then all unprefixed references to State
		 *   are to be processed. There might exist other references to State qualified by
		 *   the name of an Activity, but these need not be processed. The if branch is executed.
		 * - if the user did not select a Subsystem, they either selected an Activity
		 *   (in which case the else-if branch will possibly be entered, with no effect
		 *   at all), or they selected a component higher up in the hierarchy, which
		 *   means any reference to State to be processed is going to be prefixed. In the
		 *   latter scenario, the else-if branch is executed, possibly yielding an effect.
		 */
		if (!unprefixedStateIndices.isEmpty()) {
			for (final var index : unprefixedStateIndices) {
				property = property.substring(0, index) + modelLibrary.getMissionStateMachine(selectedPartUsage).getName() + dot + property.substring(index);
			}
		} else if (!statePrefixes.isEmpty()) {
			for (final var prefix : statePrefixes) {
				final Optional<PartUsage> matchingSubsystem = subsystemsInstances.stream()
						.filter(subsystem -> subsystem.getName().equals(prefix))
						.findAny();
				if (matchingSubsystem.isPresent()) {
					final PartUsage subsystem = matchingSubsystem.get();
					property = property.replace(dot + prefix + dotState, dot + prefix + dot + modelLibrary.getMissionStateMachine(subsystem).getName() + dotState);
				}
			}
		}
		return property;
	}

	/** {@inheritDoc} */
	@Override
	protected void okPressed()
	{
		faultModel = fault.getSelection();
		function.setTimed(timedDomainCheckBox.getSelection());

		if (isExplodtwin) {
			final String processedProperty = processProperty(propertyText.getText());
			propertyText.setText(processedProperty);
		}

		super.okPressed();
	}

	public boolean isFaultModel() {
		return faultModel;
	}

}
