/*
 * SPDX-FileCopyrightText: 2025 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.ui.dialogs;

import java.net.URL;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.osgi.framework.Bundle;

import eu.fbk.eclipse.standardtools.utils.ui.utils.DirectoryUtil;
import eu.fbk.tools.adapter.xsap.XsapFunction.AlgorithmType;

/**
 * A class to collect the parameters to run the observables optimization.
 *
 */
public class OptimizeObservablesParametersDialog extends Dialog {
	private static final String PLUGIN_ID = "eu.fbk.eclipse.explodtwin";
	private Combo algorithmTypeCombo;
	private Spinner boundLengthSpinner;

	private String[] algorithmTypes = { "bmc", "bmc_ic3"};

	/** The go ahead. */
	private boolean goAhead = false;
	private int boundLength;
	private AlgorithmType algorithmType;
	private Text alarmFileText;
	private Text observablesFileText;
	private String alarmFile;
	private String observablesFile;
	private Shell shell;
	
	/**
	 * Create the dialog.
	 *
	 * @param parentShell
	 * @param observableList 
	 */
	public OptimizeObservablesParametersDialog(Shell parentShell) {
		super(parentShell);
		this.shell = parentShell;
		setShellStyle(SWT.TITLE | SWT.RESIZE);
	}

	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("Optimize Observables Parameters");
	}

	@Override
	protected void okPressed() {
		algorithmType = AlgorithmType.valueOf(algorithmTypeCombo.getItems()[algorithmTypeCombo.getSelectionIndex()]);
		boundLength = boundLengthSpinner.getSelection();
		alarmFile = alarmFileText.getText();
		observablesFile = observablesFileText.getText();

		goAhead = true;

		super.okPressed();
	}

	@Override
	protected void cancelPressed() {
		goAhead = false;

		super.cancelPressed();
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		parent.setToolTipText("");
		final Composite container = (Composite) super.createDialogArea(parent);

		final Composite dialogContainer = new Composite(container, SWT.NONE);
		dialogContainer.setLayout(new GridLayout(3, false));
		dialogContainer.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final Label title1 = new Label(dialogContainer, SWT.NONE);
		final FontDescriptor boldDescriptor = FontDescriptor.createFrom(title1.getFont()).setStyle(SWT.BOLD);
		final Font boldFont = boldDescriptor.createFont(title1.getDisplay());
		title1.setFont(boldFont);
		title1.setText("Select the configuration files:");
		title1.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 3, 1));

		String filesPath = null;
		try {
			filesPath = DirectoryUtil.getInstance().getCurrentProjectDir();
		} catch (Exception e) {
			e.printStackTrace();
		}
		alarmFileText = prepareConfigPathGraphicalComponents(dialogContainer, "Select the alarm file", filesPath, "*.asl");
		observablesFileText = prepareConfigPathGraphicalComponents(dialogContainer, "Select the observables file",
				filesPath, "*.obs");

		final Label emptyLine = new Label(dialogContainer, SWT.HORIZONTAL);
		emptyLine.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 3, 1));

		final Label title2 = new Label(dialogContainer, SWT.NONE);
		title2.setFont(boldFont);
		title2.setText("Configure the engine:");
		title2.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 3, 1));
		
		final Label lblAlgorithm = new Label(dialogContainer, SWT.NONE);
		lblAlgorithm.setText("Algorithm Type");
		lblAlgorithm.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
		algorithmTypeCombo = new Combo(dialogContainer, SWT.NONE | SWT.READ_ONLY);
		algorithmTypeCombo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		setComboValues(algorithmTypeCombo, algorithmTypes);
		algorithmTypeCombo.select(0);

		final Label lblBoundLength = new Label(dialogContainer, SWT.NONE);
		lblBoundLength.setText("Bound Length");
		lblBoundLength.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
		boundLengthSpinner = new Spinner(dialogContainer, SWT.BORDER);
		boundLengthSpinner.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		boundLengthSpinner.setMinimum(0);
		boundLengthSpinner.setMaximum(100);
		boundLengthSpinner.setSelection(10);
		boundLengthSpinner.setIncrement(1);
		boundLengthSpinner.setPageIncrement(10);

		return container;
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
		createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
	}

	@Override
	protected Point getInitialSize() {
		return new Point(500, 300);
	}

	private void setComboValues(Combo combo, String[] admittedValues) {
		combo.setItems(admittedValues);
	}

	private Text prepareConfigPathGraphicalComponents(Composite container, String labelText, final String defaultPath,
			String extension) {
		final Bundle bundle = Platform.getBundle(PLUGIN_ID);
		final URL urlBrowseImage = FileLocator.find(bundle, new Path("icons/elipses.gif"), null);
		final Image browseImage = ImageDescriptor.createFromURL(urlBrowseImage).createImage();

		final Label confFilePathLabel = new Label(container, SWT.NONE);
		confFilePathLabel.setText(labelText);

		final Button browseConfFilePath = new Button(container, SWT.PUSH);
		browseConfFilePath.setImage(browseImage);
		browseConfFilePath.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));

		final Text confFilePath = new Text(container, SWT.SINGLE);
		confFilePath.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1));
		GridData d = new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1);
		d.widthHint = 250;
		confFilePath.setLayoutData(d);

		browseConfFilePath.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				switch (e.type) {
				case SWT.Selection:
					try {
						String pathFromDialog = getFileNameFromDialog("Choose the file", defaultPath, extension);
						if (pathFromDialog != null) {
							confFilePath.setText(pathFromDialog);
						}
					} catch (Exception e1) {
						e1.printStackTrace();
					}
					break;
				}
			}
		});

		return confFilePath;
	}

	private String getFileNameFromDialog(String title, String defaultPath, String filterExtension) throws Exception {
		final FileDialog fileDialog = new FileDialog(shell);
		fileDialog.setText(title);
		fileDialog.setFilterPath(defaultPath);
		if(filterExtension != null) {
			final String[] extensions = {filterExtension};
			fileDialog.setFilterExtensions(extensions);
		}
		final String directoryName = fileDialog.open();

		return directoryName;
	}

	public boolean goAhead() {
		return goAhead;
	}

	public int getBoundLength() {
		return boundLength;
	}

	public String getAlgorithmType() {
		return algorithmType.toString();
	}

	public String getObservablesPath() {
		return observablesFile;
	}

	public String getAlarmPath() {
		return alarmFile;
	}

}
