/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Luca Cristoforetti - initial API and implementation
 */
package eu.fbk.eclipse.explodtwin.util;

import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Function;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.XtextResourceSet;

import eu.fbk.eclipse.explodtwin.api.core.SysMLv2StateMachineModel;
import eu.fbk.eclipse.explodtwin.api.core.SysMLv2SystemModel;
import eu.fbk.eclipse.explodtwin.api.util.ModelUtil;
import eu.fbk.eclipse.explodtwin.ui.dialogs.ElementSelectorDialog;
import eu.fbk.eclipse.standardTools.XSapExecService.services.XSapExecService;
import eu.fbk.eclipse.standardtools.ModelTranslatorToOcra.ui.services.OSSTranslatorServiceUI;
import eu.fbk.eclipse.standardtools.StateMachineTranslatorToSmv.core.services.FeiTranslatorServiceAPI;
import eu.fbk.eclipse.standardtools.StateMachineTranslatorToSmv.core.services.SMVTranslatorServiceAPI;
import eu.fbk.eclipse.standardtools.StateMachineTranslatorToSmv.core.utils.SmvModelUtil;
import eu.fbk.eclipse.standardtools.nuXmvService.ui.utils.NuXmvDirectoryUtil;
import eu.fbk.eclipse.standardtools.utils.core.utils.FileSystemUtil;
import eu.fbk.eclipse.standardtools.utils.core.utils.Pair;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.DirectoryUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.ErrorsDialogUtil;
import eu.fbk.eclipse.standardtools.utils.ui.utils.OCRADirectoryUtil;
import eu.fbk.eclipse.standardtools.xtextService.core.utils.XTextResourceUtil;
import eu.fbk.sysmlv2.sysMLv2.ConnectionElement;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Part;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.QualifiedNameExpression;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.Usage;
import eu.fbk.sysmlv2.util.SysMLv2Util;
import eu.fbk.tools.adapter.ui.commands.contract.PrintSystemImplementationCommand;
import eu.fbk.tools.adapter.ui.preferences.PreferenceConstants;
import eu.fbk.tools.editor.basetype.baseType.Expression;
import eu.fbk.tools.editor.nusmv.smv.ComplexIdentifier;
import eu.fbk.tools.editor.nusmv.smv.DefineBody;
import eu.fbk.tools.editor.nusmv.smv.DefineDeclaration;
import eu.fbk.tools.editor.nusmv.smv.FunBody;
import eu.fbk.tools.editor.nusmv.smv.FunctionDeclaration;
import eu.fbk.tools.editor.nusmv.smv.IVariableDeclaration;
import eu.fbk.tools.editor.nusmv.smv.Model;
import eu.fbk.tools.editor.nusmv.smv.Module;
import eu.fbk.tools.editor.nusmv.smv.ModuleElement;
import eu.fbk.tools.editor.nusmv.smv.ModuleParameter;
import eu.fbk.tools.editor.nusmv.smv.ModuleType;
import eu.fbk.tools.editor.nusmv.smv.SmvFactory;
import eu.fbk.tools.editor.nusmv.smv.TimeAnnotation;
import eu.fbk.tools.editor.nusmv.smv.VarBody;
import eu.fbk.tools.editor.nusmv.smv.VariableDeclaration;

public class AdapterUtil {
	private static final String SM_SUFFIX = "_SM";
	private static final String MAIN = "main";
	private static final SmvFactory SMV_FACTORY = SmvFactory.eINSTANCE;

	/**
	 * This record contains the names of artifacts used for extending the model with faults
	 */
	public record GeneratedFiles(String modelName, String feiFileName, String expandedFeiFileName,
			String extendedSmvFileName, String fmsFileName, String ftFileName) {}

    private static final String ID_PARAMETER_MAP_FILE_PATH = "map_file";
    private static final String ID_PARAMETER_MODEL_FILE_OSS_PATH = "contract_model";
    private static final String ID_PARAMETER_MODEL_TIME = "time_model";
    private static final String ID_PARAMETER_RESULT_FILE = "result_file";
    private static final String ID_PARAMETER_USE_OLD_MODEL = "old_model_format";
    private static final String ID_INTERNAL_EXECUTION = "internal_execution";
    private static final String ID_DISABLE_ASYNC_CONSTRAINTS = "disable_async_constraints";
    private static final String SMV_EXT = ".smv";
    private static final String FEI_EXT = ".fei";
    private static final String XML_EXT = ".xml";
    private static final String RESULTS_FILE_PATH = File.separator + "VerificationResults";
    private static final SysMLv2SystemModel SYSTEM_MODEL = new SysMLv2SystemModel();
    private static final SysMLv2StateMachineModel STATE_MACHINE_MODEL = new SysMLv2StateMachineModel();

	/**
	 * Given a SysML file, returns the Xtext resource.
	 * @param iFile
	 * @return
	 */
	public static Resource getResource(IFile iFile) {
		if (iFile.getFileExtension().equalsIgnoreCase("sysml")) {
			XtextResourceSet resourceSet = new XtextResourceSet();
			resourceSet.addLoadOption(XtextResource.OPTION_ENCODING, "UTF-8");
			Resource resource = resourceSet.getResource(URI.createFileURI(iFile.getLocation().toOSString()), true);
			if (resource == null) {
				return null;
			}
			return resource;
		}

		return null;
	}

	/**
	 * Creates or loads the project.
	 *
	 * @param projectName
	 * @return
	 * @throws ExecutionException
	 */
	public static IProject initProject(String projectName) throws ExecutionException {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		IProgressMonitor monitor = new NullProgressMonitor();

		if (project == null) {
			throw new ExecutionException("Can't get project '" + projectName + "' from workspace.");
		}

		if (project.exists()) {
			try {
				project.delete(true, monitor);
			} catch (CoreException e) {
				throw new ExecutionException("Can't delete project '" + projectName + "'.", e);
			}
		}

		try {
			project.create(monitor);
		} catch (CoreException e) {
			throw new ExecutionException("Can't create project '" + projectName + "'.", e);
		}

		if (!project.isOpen()) {
			try {
				project.open(null);
			} catch (CoreException e) {
				throw new ExecutionException("Can't open project '" + projectName + "'.", e);
			}
		}

		return project;
	}

	public static String generateOCRA(PartUsage systemComponent, int timeSpecification) {
		OSSTranslatorServiceUI ossTranslatorService = OSSTranslatorServiceUI.getInstance(SYSTEM_MODEL);
		String workspaceDir =
				eu.fbk.tools.adapter.ui.Activator.getDefault().getPreferenceStore().getString(PreferenceConstants.TOOL_WORKSPACE);
		boolean showPopups = false;
		boolean isAsyncCommunication = true;	// This is not harmful in case no components are declared aysnc
		boolean useXtextValidation = true;

		System.out.println("workspaceDir = " + workspaceDir);

		try {
			File ossFile = ossTranslatorService.exportRootComponentToOssFile(systemComponent, systemComponent.eResource(), timeSpecification,
					isAsyncCommunication, useXtextValidation, showPopups, workspaceDir, new NullProgressMonitor());
			System.out.println("OSS file = " + ossFile.getAbsolutePath());
			return ossFile.getAbsolutePath();
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Generates the SMV file for the given element. If the element is the root, a monolithic file will
	 * be generated, containing all the state machines. It requires that the OCRA tool is available.
	 * @param selectedPart the element to export in SMV
	 * @param timeSpecification
	 * 				0 for hybrid model, 1 for discrete model, 2 for timed model
	 * @return the path of the SMV file
	 */
	public static String generateSMV(PartUsage selectedPart, int timeSpecification) {
		String workspaceDir =
				eu.fbk.tools.adapter.ui.Activator.getDefault().getPreferenceStore().getString(PreferenceConstants.TOOL_WORKSPACE);
		OCRADirectoryUtil ocraDirectoryUtil = OCRADirectoryUtil.getInstance();
		NuXmvDirectoryUtil nuXmvDirectoryUtil = NuXmvDirectoryUtil.getInstance();
		OSSTranslatorServiceUI ossTranslatorServiceUI = OSSTranslatorServiceUI.getInstance(SYSTEM_MODEL);
		IProgressMonitor monitor = new NullProgressMonitor();
		boolean showPopups = false;
		boolean internalExecution = true;
		boolean isAsyncCommunication = true; // This is not harmful in case no components are declared aysnc
		boolean useXtextValidation = true;
		boolean disableAsyncConstraints = false;
		String default_USE_OLD_MODEL = "false";
		String model_time = null;

		//TODO: this part should be rewritten to avoid using the OCRA tool to create the monolithic SMV file
		// Export the whole model or a single component
		try {
			if (SysMLv2Util.isSystemPartUsage(selectedPart)) {
				// Export all the components to single SMV files and populate the linking map
				// The library method cannot be used because it will export only one state machine per component.
				HashMap<String, String> smvPathComponentNameMap = new HashMap<>();
				for (PartDefinition component : SYSTEM_MODEL.getComponents(selectedPart)) {
					String smvFile = GenerateSingleSMV(component, workspaceDir, monitor, timeSpecification);
					if (smvFile != null) {
						String compName = SYSTEM_MODEL.getComponentName(component);
						smvPathComponentNameMap.put(smvFile, compName);
					}
				}

				String ossDirPath = ocraDirectoryUtil.getOSSDirPath();
				String smvMapDirPath = nuXmvDirectoryUtil.getSmvFileDirectory();
				String monolithicSMVFilePath = nuXmvDirectoryUtil.getMonolithicSMVFilePath(selectedPart.getName());

				System.out.println("ossDirPath = " + ossDirPath);
				System.out.println("smvMapDirPath = " + smvMapDirPath);
				System.out.println("monolithicSMVFilePath = " + monolithicSMVFilePath);

				// Create a file with the map of the SMVs
				File smvMapFile = FileSystemUtil.createSmvMapFile("map", smvMapDirPath, smvPathComponentNameMap);
				System.out.println("smvMapFile = " + smvMapFile);

				// Export the OSS of the root component
				File ossFile = ossTranslatorServiceUI.exportRootComponentToOssFile(selectedPart, selectedPart.eResource(),
						timeSpecification, isAsyncCommunication, useXtextValidation, showPopups, ossDirPath, monitor);
				switch (timeSpecification) {
				case 0:
					model_time = "hybrid";
					break;
				case 1:
					model_time = "discrete";
					break;
				case 2:
					model_time = "timed";
					break;
				}

				// Prepare parameters
				Map<String, String> parameters = Map.of(
						ID_PARAMETER_MODEL_FILE_OSS_PATH, ossFile.getPath(),
						ID_PARAMETER_MAP_FILE_PATH, smvMapFile.getPath(),
						ID_PARAMETER_MODEL_TIME, model_time,
						ID_PARAMETER_USE_OLD_MODEL, default_USE_OLD_MODEL,
						ID_INTERNAL_EXECUTION, String.valueOf(internalExecution),
						ID_PARAMETER_RESULT_FILE, monolithicSMVFilePath,
						ID_DISABLE_ASYNC_CONSTRAINTS, String.valueOf(disableAsyncConstraints));


	            // Create a minimal ExecutionEvent
	            ExecutionEvent event = new ExecutionEvent(null, parameters, null, null);

	            // Execute the EATA command directly
				PrintSystemImplementationCommand cmd = new PrintSystemImplementationCommand();
				try {
		            cmd.execute(event);
		        } catch (ExecutionException e) {
		            e.printStackTrace();
		        }
				return monolithicSMVFilePath;
			} else {
				return GenerateSingleSMV(selectedPart, workspaceDir, monitor, timeSpecification);
			}
		} catch (Exception e) {
        	DialogUtil.getInstance().showMessage_ExceptionError(e);
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Generates a SMV for a component. If the component is a subsystem, the file should incorporate
	 * also the state machines of the activities.
	 * @param component
	 * @param workspaceDir
	 * @param monitor
	 * @param timeSpecification
	 * @return
	 * @throws Exception
	 */
	private static String GenerateSingleSMV(Part component, String workspaceDir,
			IProgressMonitor monitor, int timeSpecification) throws Exception {
		File smvFile =  null;
		SMVTranslatorServiceAPI smvTranslatorServiceAPI = SMVTranslatorServiceAPI.getInstance(SYSTEM_MODEL, STATE_MACHINE_MODEL);

		if (ModelUtil.isSubsystem(component)) {
			// Export all the state machines on separate SMV files
			Map<String, String> smvPathComponentNameMap = new HashMap<>();
			Set<StateUsage> stateMachines = STATE_MACHINE_MODEL.getNominalStateMachinesIncludingFromSubComponents(component);
			for (StateUsage stateMachine : stateMachines) {
				String fsmName = STATE_MACHINE_MODEL.getStateMachineName(stateMachine);
				smvFile = smvTranslatorServiceAPI.exportStateMachineToSmvFile(stateMachine, workspaceDir, fsmName, monitor, timeSpecification);
				smvPathComponentNameMap.put(smvFile.getPath(), fsmName);
				System.out.println("Storing " + fsmName + " in " + smvFile.getPath());
			}

			// Retrieve the mission state machine
			StateUsage missionStateMachine = ModelUtil.getMissionStateMachine(component);

		    String timeAnnotation = "continuous";
			return composeSMV(smvPathComponentNameMap, workspaceDir, monitor, missionStateMachine, timeAnnotation, component);
		} else {

			// Regular component
			Object stateMachine = STATE_MACHINE_MODEL.getFirstNominalStateMachine(component);
			if (stateMachine != null) {
				String compName = SYSTEM_MODEL.getComponentName(component);
				smvFile = smvTranslatorServiceAPI.exportStateMachineToSmvFile(stateMachine, workspaceDir, compName, monitor, timeSpecification);
			} else {
				return null;
			}
		}

		return (smvFile != null) ? smvFile.getPath() : null;
	}

	/**
	 * Compose a single SMV using the given main and submodules.
	 * @param smvPathComponentNameMap the map with all the SMVs
	 * @param workspaceDir
	 * @param monitor
	 * @param missionStateMachine
	 * @param timeSpecification
	 * @param component
	 * @return the path of the SMV file
	 * @throws Exception
	 */
	private static String composeSMV(Map<String, String> smvPathComponentNameMap, String workspaceDir,
			IProgressMonitor monitor, StateUsage missionStateMachine, String timeSpecification, Part component)
			throws Exception {
		XTextResourceUtil xtextResourceUtil = XTextResourceUtil.getInstance();
		List<Module> subModules = new ArrayList<Module>();
		List<Module> mains = new ArrayList<Module>();
		String compInstanceName = SYSTEM_MODEL.getComponentName(component);
		String compTypeName = SYSTEM_MODEL.getComponentTypeName(component);	// This is the owner of mission, it could be different from component
		String compModuleName = (component instanceof PartUsage) ? compTypeName : compInstanceName;

		String missionName = STATE_MACHINE_MODEL.getStateMachineName(missionStateMachine);

		// Read all the SMV files, parse them and extract the component modules and main modules
		Module moduleSubsysModule = null; // This is the component module of the subsystem
		for (Entry<String, String> entry : smvPathComponentNameMap.entrySet()) {
			String content = "";
		    try {
		        content = Files.readString(Paths.get(entry.getKey()));
		    } catch (IOException e) {
		        System.err.println("Error reading file: " + e.getMessage());
		    }
		    Model smvModel = xtextResourceUtil.deserializeSmvModel(content);

		    for (Module module : smvModel.getModules()) {
		    	if (module.getIdentifier().equals(MAIN)) {
		    		mains.add(module);
		    	} else {
		    		subModules.add(module);
					if (module.getIdentifier().equals(compTypeName)) {
						moduleSubsysModule = EcoreUtil.copy(module);
					}
		    	}
		    }
		}

		// Collect the name of the instances from the model, with their corresponding types
		Map<String, String> instances = new HashMap<>();
		for (PartUsage instance : typeSelect(ModelUtil.getComponentType(component).getMembers(), PartUsage.class)) {
			instances.put(SYSTEM_MODEL.getName(instance), SYSTEM_MODEL.getComponentInstanceTypeName(instance));
		}

		//TODO: only the mission state machine is added to the component's table. Also platform should appear
		//FIXME: The submodule is named as the component, but it should somehow be customized
		instances.put(missionName, compTypeName);

		// Extract all the module instances declarations and the whole main of mission
		// Also extract function declarations
		Map<String, VarBody> moduleInstances = new HashMap<>();
		Map<String, EList<FunBody>> functions = new HashMap<>();
		Module mainSubsysModule = null;	// This is the mission module
		for (Module module : mains) {
			for (VarBody variable : extractVarBodies(module)) {
				if (variable.getType() instanceof ModuleType) {
					String moduleType = variable.getIdentifier().getIdentifier().getValue();

					// Extract the functions of the module and store the list in a map
					EList<FunBody> moduleFunctions = new BasicEList<FunBody>();
					for (FunBody function : extractFunBodies(module)) {
						moduleFunctions.add(EcoreUtil.copy(function));
					}
					functions.put(moduleType, moduleFunctions);

					// Store the module instances
					moduleInstances.put(moduleType, EcoreUtil.copy(variable));

					// Store the main module of the mission
					if (variable.getIdentifier().getIdentifier().getValue().equals(compTypeName)) {
						mainSubsysModule = EcoreUtil.copy(module);
					}
				}
			}
		}

		// A map that describes the connections among modules actual parameters
		Map<String,String> connectionsMap = new HashMap<>();
		for (ConnectionElement connection : ModelUtil.getConnectionsPorts(component)) {
			Object connSource = SYSTEM_MODEL.getConnectorSource(connection);
			String sourceEndName = SYSTEM_MODEL.getConnectorEndName(connSource);
			Object connTarget = SYSTEM_MODEL.getConnectorTarget(connection);
			String targetEndName = SYSTEM_MODEL.getConnectorEndName(connTarget);

			boolean isOut = false;
			if (connSource instanceof QualifiedNameExpression expr) {
				isOut = ModelUtil.isOutputParameter(expr.getReferencedElement());
			}

			String prefix = getConnectorOwnerName(SYSTEM_MODEL.getConnectorEndOwner(connSource));
			if(isOut && prefix == null) {
				prefix = missionName;
			}
			String source = prefix == null ? sourceEndName : prefix + "." + sourceEndName;

			boolean isIn = false;
			if (connTarget instanceof QualifiedNameExpression expr) {
				isIn = ModelUtil.isInputParameter(expr.getReferencedElement());
			}
			prefix = getConnectorOwnerName(SYSTEM_MODEL.getConnectorEndOwner(connTarget));
			if(isIn && prefix == null) {
				prefix = missionName;
			}
			String target = prefix == null ? targetEndName : prefix + "." + targetEndName;
			connectionsMap.put(target, source);
		}

		// This is a list containing internal vars that should not appear in the component module
		List<String> internalVars = computeInternalVar(component, missionStateMachine, SYSTEM_MODEL, STATE_MACHINE_MODEL);

		// Create the module for the subsystem
		Module subsystemModule = SMV_FACTORY.createModule();
		subsystemModule.setIdentifier(compModuleName);
		subsystemModule.getModuleParameters().addAll(moduleSubsysModule.getModuleParameters());
		subsystemModule = removeInternalVarsFromModule(subsystemModule, internalVars);

		EList<FunBody> qualifiedFunctions = new BasicEList<FunBody>();

		// For each instance, create a VAR declaration with its correct type and store it
		VariableDeclaration varDeclaration = SMV_FACTORY.createVariableDeclaration();
		for (String instanceName : instances.keySet()) {
			VarBody varBody = EcoreUtil.copy(moduleInstances.get(instances.get(instanceName)));
			varBody.getIdentifier().getIdentifier().setValue(instanceName);

			// Qualify the functions with the name of the correct instance
			EList<FunBody> moduleFunctions = functions.get(instances.get(instanceName));
			moduleFunctions = qualifyFunctions(moduleFunctions, instanceName);
			qualifiedFunctions.addAll(moduleFunctions);

			// Create an instance with the correct linkings
			VarBody moduleInstance = createInstance(instanceName, varBody, connectionsMap, compTypeName, moduleFunctions);
			varDeclaration.getVariables().add(moduleInstance);
		}
		subsystemModule.getModuleElements().add(varDeclaration);

		/*
		 * The following code populates the DEFINE section of each subsystem's module.
		 * Variables in the DEFINE section represent the module outputs.
		 * We must modify the connections between the module outputs and the submodule
		 * ports they are backed by, in order to make sure that each output is connected
		 * to the correct submodule port (e.g. setter ports relaying a setter towards another
		 * subsystem must be bound to the setter origin port in the correct activity).
		 */
		EList<DefineBody> defineBodies = renameDefineBodies(extractDefineBodies(mainSubsysModule), missionName, connectionsMap);
		if (!defineBodies.isEmpty()) {
			DefineDeclaration defines = SMV_FACTORY.createDefineDeclaration();
			defines.getBodies().addAll(defineBodies);
			subsystemModule.getModuleElements().add(defines);
		}

		// Add the FUN section if FUN are present in submodules
		if (!qualifiedFunctions.isEmpty()) {
			FunctionDeclaration functionDeclarations = SMV_FACTORY.createFunctionDeclaration();
			functionDeclarations.getVariables().addAll(qualifiedFunctions);
			subsystemModule.getModuleElements().add(functionDeclarations);
		}

		//TODO: can other sections be necessary? In general, also FROZENVAR and CONSTANTS sections could be preserved

		// Create the main module
		Module mainModule = SMV_FACTORY.createModule();
		mainModule.setIdentifier(MAIN);

		// Create an instance of the component
		VariableDeclaration mainVarDeclaration = SMV_FACTORY.createVariableDeclaration();
		VarBody varBody = EcoreUtil.copy(moduleInstances.get(compModuleName));
		varBody.getIdentifier().getIdentifier().setValue(compModuleName);

		// Internal variables should be removed from actual parameters in the component module
		varBody = removeInternalVarsFromParameters(varBody, internalVars);

		mainVarDeclaration.getVariables().add(varBody);
		mainModule.getModuleElements().add(mainVarDeclaration);

		// Add the VAR section excluding module instances and internal vars
		EList<VarBody> varBodies = extractVarBodies(mainSubsysModule);
		varBodies = removeInternalVarsFromVars(varBodies, internalVars);
		VariableDeclaration mainVars = SMV_FACTORY.createVariableDeclaration();
		for (VarBody body : varBodies) {
			if (!(body.getType() instanceof ModuleType)) {
				mainVars.getVariables().add(body);
			}
		}
		if (!mainVars.getVariables().isEmpty()) {
			mainModule.getModuleElements().add(mainVars);
		}

		// Add the IVAR section, avoiding internal vars
		EList<VarBody> ivarBodies = extractIvarBodies(mainSubsysModule);
		ivarBodies = removeInternalVarsFromVars(ivarBodies, internalVars);
		if (!ivarBodies.isEmpty()) {
			IVariableDeclaration mainIvars = SMV_FACTORY.createIVariableDeclaration();
			mainIvars.getVariables().addAll(ivarBodies);
			mainModule.getModuleElements().add(mainIvars);
		}

		// Add the DEFINE section. It represents the output variables of the module. Module name should be adapted (but maybe already correct)
		EList<DefineBody> mainDefineBodies = renameDefineBodies(extractDefineBodies(mainSubsysModule), compModuleName, Map.of());
		if (!mainDefineBodies.isEmpty()) {
			DefineDeclaration mainDefines = SMV_FACTORY.createDefineDeclaration();
			mainDefines.getBodies().addAll(mainDefineBodies);
			mainModule.getModuleElements().add(mainDefines);
		}

		// Compose the new SMV model
		Model finalModel = xtextResourceUtil.deserializeSmvModel("MODULE dummy\n");
		finalModel.getModules().clear();
		TimeAnnotation timeAnnotation = SMV_FACTORY.createTimeAnnotation();
		timeAnnotation.setValue(timeSpecification);
		finalModel.setAnnotation(timeAnnotation);
		finalModel.getModules().add(mainModule);
		finalModel.getModules().add(subsystemModule);
		for (Module module : subModules) {
			if (module.getIdentifier().equals(moduleSubsysModule.getIdentifier())) {
				module = changeModuleName(module);
			}
			finalModel.getModules().add(EcoreUtil.copy(module));
		}

		File smvFile = SmvModelUtil.getInstance().generateSmvFileFromSmvModel(workspaceDir, compInstanceName, finalModel, monitor);
		return smvFile.getPath();
	}

	/**
	 * Qualify the list of functions adding the given prefix.
	 * @param moduleFunctions
	 * @param funPrefix
	 * @return
	 * @throws Exception
	 */
	private static EList<FunBody> qualifyFunctions(EList<FunBody> moduleFunctions, String funPrefix) throws Exception {
		EList<FunBody> qualifiedFunctions = new BasicEList<FunBody>();
		for (FunBody function : moduleFunctions) {
			Expression expr = createSMVExpression("_" + funPrefix + "." + function.getIdentifier().getIdentifier().getValue());
			FunBody qualified = EcoreUtil.copy(function);
			qualified.setIdentifier((ComplexIdentifier) expr);
			qualifiedFunctions.add(qualified);
		}
		return qualifiedFunctions;
	}

	/**
	 * Compute the list of variables that were added to connect the activities.
	 * @param component
	 * @param missionStateMachine
	 * @param systemModel
	 * @param stateMachineModel
	 * @return
	 */
	private static List<String> computeInternalVar(EObject component, StateUsage missionStateMachine,
			SysMLv2SystemModel systemModel, SysMLv2StateMachineModel stateMachineModel) {
		List<String> internalVars = new ArrayList<>();

		// Get the ports from AbstractSystemModel, used for OCRA export
		EList <DirectedElement> inports = systemModel.getNonStaticInputPorts(component);

		// Get the ports from AbstractStateMachineModel, used for SMV export
		EList <DirectedElement> statePorts = stateMachineModel.getOwnerInputPortsExceptEvents(missionStateMachine);
		statePorts.addAll(stateMachineModel.getOwnerInputEvents(missionStateMachine));

		statePorts.removeAll(inports);
		for (DirectedElement directedElement : statePorts) {
			internalVars.add(directedElement.getName());
		}
		return internalVars;
	}

	private static Module removeInternalVarsFromModule(Module module, List<String> internalVars) {
		Module updatedModule = EcoreUtil.copy(module);
		updatedModule.getModuleParameters().clear();
		for (ModuleParameter parameter : module.getModuleParameters()) {
			if (internalVars.contains(parameter.getIdentifier())) {
				continue;
			} else {
				updatedModule.getModuleParameters().add(EcoreUtil.copy(parameter));
			}
		}
		return updatedModule;
	}

	/**
	 * Remove the vars that belong to the list of internal variables.
	 * @param varBodies
	 * @param internalVars
	 * @return
	 */
	private static EList<VarBody> removeInternalVarsFromVars(EList<VarBody> varBodies, List<String> internalVars) {
		EList<VarBody> updatedList = new BasicEList<VarBody>();
		for (VarBody varBody : varBodies) {
			if (internalVars.contains(varBody.getIdentifier().getIdentifier().getValue())) {
				continue;
			} else {
				updatedList.add(EcoreUtil.copy(varBody));
			}
		}
		return updatedList;
	}

	/**
	 * Removes the parameters that belong to the list of internal variables.
	 * @param varBody
	 * @param internalVars
	 * @return
	 */
	private static VarBody removeInternalVarsFromParameters(VarBody varBody, List<String> internalVars) {
		VarBody updatedVar = EcoreUtil.copy(varBody);
		ModuleType type = (ModuleType) updatedVar.getType();
		type.getParams().clear();
		for (Expression param : ((ModuleType) varBody.getType()).getParams()) {
			if (param instanceof ComplexIdentifier identifier) {
				if (internalVars.contains(identifier.getIdentifier().getValue())) {
					continue;
				} else {
					type.getParams().add(EcoreUtil.copy(param));
				}
			}
		}
		return updatedVar;
	}

	/**
	 * Add a suffix to the module name, to avoid clashing with component name.
	 * @param module
	 * @return
	 */
	private static Module changeModuleName(Module module) {
		Module renamed = EcoreUtil.copy(module);
		renamed.setIdentifier(module.getIdentifier() + SM_SUFFIX);
		return renamed;
	}

	private static <T extends EObject, R> EList<R> extractBodies(Module module, Class<T> declarationClass, Function<T, EList<R>> extractor) {
	    EList<R> bodies = new BasicEList<>();
	    for (ModuleElement element : module.getModuleElements()) {
	        if (declarationClass.isInstance(element)) {
	            T declaration = declarationClass.cast(element);
	            bodies.addAll(extractor.apply(declaration));
	        }
	    }
	    return bodies;
	}

	private static EList<DefineBody> extractDefineBodies(Module subSysModule) {
	    return extractBodies(subSysModule, DefineDeclaration.class, DefineDeclaration::getBodies);
	}

	private static EList<VarBody> extractVarBodies(Module subSysModule) {
	    return extractBodies(subSysModule, VariableDeclaration.class, VariableDeclaration::getVariables);
	}

	private static EList<VarBody> extractIvarBodies(Module subSysModule) {
	    return extractBodies(subSysModule, IVariableDeclaration.class, IVariableDeclaration::getVariables);
	}

	private static EList<FunBody> extractFunBodies(Module subSysModule) {
	    return extractBodies(subSysModule, FunctionDeclaration.class, FunctionDeclaration::getVariables);
	}

	private static EList<DefineBody> renameDefineBodies(EList<DefineBody> bodies, String newPrefix, Map<String, String> connections) {
		EList<DefineBody> adaptedBodies = new BasicEList<DefineBody>();
		for (DefineBody body : bodies) {
			DefineBody adapted = EcoreUtil.copy(body);
			String variableName = adapted.getVariable().getIdentifier().getValue();
			if (adapted.getExpression() instanceof ComplexIdentifier complexId) {
				String[] connectionSource = connections.entrySet().stream()
					.filter(entry -> entry.getKey().equals(variableName))
					.map(Entry::getValue)
					.map(source -> source.split("\\."))
					.filter(array -> array.length == 2)
					.findAny().orElse(new String[] {newPrefix, variableName}); // fall back on the newPrefix argument if no connection is found
				complexId.getLeft().getIdentifier().setValue(connectionSource[0]);
				((ComplexIdentifier)complexId.getRight()).getIdentifier().setValue(connectionSource[1]);
			}
			adaptedBodies.add(adapted);
		}
		return adaptedBodies;
	}

	private static String getConnectorOwnerName(Object connectorEndOwner) {
		return (connectorEndOwner instanceof PartUsage usage) ? usage.getName() : null;
	}

	/**
	 * Creates an instance of a module, substituting the formal parameters with the actual ones.
	 * @param instanceName the name of the instance
	 * @param variable
	 * @param conns
	 * @param compTypeName the name of the component type, to be changed
	 * @param moduleFunctions
	 * @return
	 * @throws Exception
	 */
	private static VarBody createInstance(String instanceName, VarBody variable, Map<String, String> conns,
			String compTypeName, EList<FunBody> moduleFunctions) throws Exception {
		if (variable.getType() instanceof ModuleType moduleType) {
			ModuleType varType = SMV_FACTORY.createModuleType();
			String moduleTypeName = moduleType.getIdentifier();

			// If the module is the component one, add a suffix to avoid clashing
			if (moduleTypeName.equals(compTypeName)) {
				moduleTypeName += SM_SUFFIX;
			}
			varType.setIdentifier(moduleTypeName);
			for (Expression param : moduleType.getParams()) {
				if (param instanceof ComplexIdentifier identifier) {
					String paramName = identifier.getIdentifier().getValue();
					String identifierName = instanceName + "." + paramName;
					Expression expr = createSMVExpression(conns.get(identifierName));
					if (expr != null) {
						varType.getParams().add(expr);
					} else {	// Check if the parameter is a function
						expr = createSMVExpression(getQualifiedFunction(paramName, moduleFunctions));
						if (expr != null) {
							varType.getParams().add(expr);
						} else {	// Keep the param as it is, is a component input port
							varType.getParams().add(EcoreUtil.copy(param));
						}
					}
				} else {
					System.err.println("Type of module parameter not supported.");
				}
			}
			variable.setType(EcoreUtil.copy(varType));
			return variable;
		}
		return null;
	}

	/**
	 * Given a parameter name, returns the qualified function name, if matching.
	 * @param paramName the ipotetical name of the function
	 * @param moduleFunctions the list of actual functions
	 * @return
	 */
	private static String getQualifiedFunction(String paramName, EList<FunBody> moduleFunctions) {
		for (FunBody function : moduleFunctions) {
			if (paramName.equals(((ComplexIdentifier) function.getIdentifier().getRight()).getIdentifier().getValue())) {
				return function.getIdentifier().getLeft().getIdentifier().getValue() + "." + paramName;
			}
		}
		return null;
	}

	/**
	 * Simple method to convert a string to a SMV expression, using the deserializer.
	 * @param expression
	 * @return
	 * @throws Exception
	 */
	private static Expression createSMVExpression(String expression) throws Exception {
		if (expression != null) {

			// Create a dummy SMV file with the expression
			String modelBody = "MODULE main VAR instance : Type ( " + expression + ");";

			// Parse the model and extract the expression
			Model smvModel = XTextResourceUtil.getInstance().deserializeSmvModel(modelBody);
			List<ModuleType> varType = EcoreUtil2.getAllContentsOfType(smvModel, ModuleType.class);
			return varType.get(0).getParams().get(0);
		}
		return null;
	}

	/**
	 * Exports the model to SMV. This in an interactive method, it asks the user to select the component to export.
	 * @param resource the selected resource
	 * @param activeShell the active shell
	 * @param timeSpecification
	 * 				0 for hybrid model, 1 for discrete model, 2 for timed model
	 * @return a map with the selected component and the path of the generated SMV file
	 */
	public static Pair<Usage, String> processSysMLModel(Resource resource, Shell activeShell, int timeSpecification) {
		if (resource.getContents().size() != 1) {
			return null;
		}

		EObject rootElement = resource.getContents().get(0);

		List<PartUsage> partUsages = ModelUtil.getAllContentsOfTypeFromModel(
				EcoreUtil2.getResourceSet(rootElement), PartUsage.class);
		if (partUsages.isEmpty()) {
			return null;
		}
		partUsages.sort((part1, part2) -> part1.getName().compareTo(part2.getName()));

		// Discard the parts without state machines
		partUsages.removeIf(p -> STATE_MACHINE_MODEL.getFirstNominalStateMachine(p) == null);

		// Add the system part usage
		PartUsage system;
		try {
			system = ModelUtil.getSystemPart(EcoreUtil2.getResourceSet(rootElement));
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			return null;
		}
		partUsages.add(0, system);

		PartUsage selectedPart = getSelectedPart(activeShell, partUsages);

		if (selectedPart == null) {
			return null;
		}

		return new Pair<Usage, String>(selectedPart, generateSMV(selectedPart, timeSpecification));
	}

	/**
	 * Returns a part selected from a list, opening a dialog and asking the user.
	 * @param activeShell
	 * @param parts the list of parts to choose from
	 * @return the selected part
	 */
	public static PartUsage getSelectedPart(Shell activeShell, List<PartUsage> parts) {
		if (parts.size() == 1) {
			return parts.get(0);
		} else if (parts.size() > 1) {
			ElementSelectorDialog dialog = new ElementSelectorDialog(activeShell, "Component Selector",
					"Select a component instance:");
			dialog.addElements(parts);
			if (dialog.open() == ElementSelectorDialog.OK) {
				return (PartUsage) dialog.getSelection();
			}
		}
		return null;
	}

	/**
	 * Prepares all the needed files for the subsequent analysis.
	 * @param systemComponent
	 * @param smvFileName
	 * @return true if all the files were generated
	 * @throws Exception
	 */
	public static GeneratedFiles prepareExpandedFiles(PartUsage systemComponent, String smvFileName) throws Exception {

		// Prepare the file names for all the commands
		String fileName = SYSTEM_MODEL.getName(ModelUtil.getModel(systemComponent));
		String systemName = SYSTEM_MODEL.getName(systemComponent);
		String modelName = fileName + "_" + systemName;

		String feiFileName = computeFeiFileName(modelName);
		String expandedFeiFileName = computeExpandedFeiFileName(modelName);
		String extendedSmvFileName = computeExtendedSmvFileName(modelName);
		String fmsFileName = computeFmsFileName(modelName);
		String ftFileName = computeFtFileName(modelName);

		GeneratedFiles files = new GeneratedFiles(modelName, feiFileName, expandedFeiFileName, extendedSmvFileName,
				fmsFileName, ftFileName);

		FeiTranslatorServiceAPI.getInstance(SYSTEM_MODEL, STATE_MACHINE_MODEL)
				.exportFaultyStateMachinesToFeiFile(ModelUtil.getModel(systemComponent),
						NuXmvDirectoryUtil.getInstance().getSmvFileDirectory(), modelName, null, new NullProgressMonitor());

		// Expand the FEI file
		if (!XSapExecService.getInstance().expandFaultExtensions(feiFileName, expandedFeiFileName, true)) {
			return null;
		}

		// Extend the SMV model
		if (!XSapExecService.getInstance().extendModel(smvFileName, expandedFeiFileName, fmsFileName, extendedSmvFileName, true)) {
			return null;
		}
		return files;
	}

	private static String computeFmsFileName(String modelName) throws Exception {
		return NuXmvDirectoryUtil.getInstance().getSmvTempDirectory() + File.separator + "fms_" + modelName + XML_EXT;
	}

	private static String computeExtendedSmvFileName(String modelName) throws Exception {
		return NuXmvDirectoryUtil.getInstance().getSmvTempDirectory() + File.separator + "extended_" + modelName + SMV_EXT;
	}

	private static String computeExpandedFeiFileName(String modelName) throws Exception {
		return NuXmvDirectoryUtil.getInstance().getSmvTempDirectory() + File.separator + "expanded_" + modelName + XML_EXT;
	}

	private static String computeFeiFileName(String modelName) throws Exception {
		return NuXmvDirectoryUtil.getInstance().getSmvFileDirectory() + File.separator + modelName + FEI_EXT;
	}

	private static String computeFtFileName(String modelName) throws Exception {
		return getResultDir() + File.separator + "extended_" + modelName + "_" + getDate() + XML_EXT;
	}

	/**
	 * Gives the current date and time formatted in a certain way.
	 *
	 * @return the string representing the current date and time
	 */
	private static String getDate() {
		String pattern = "yyyy-MM-dd-HH-mm-ss";
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
		return simpleDateFormat.format(new Date());
	}

	/**
	 * Returns the directory where validation result files are stored. If the
	 * directory is not present, it will be created.
	 *
	 * @return the string containing the directory
	 * @throws Exception
	 */
	private static String getResultDir() throws Exception {
		String resultDir = DirectoryUtil.getInstance().getCurrentProjectDir() + RESULTS_FILE_PATH;
		File directory = new File(resultDir);
		if (!directory.exists()) {
			return createResultDir(resultDir);
		} else {
			return resultDir;
		}
	}

	/**
	 * Creates the directory where to store validation results.
	 *
	 * @return the string containing the directory
	 */
	private static String createResultDir(String dirName) {
		try {
			File directory = new File(dirName);
			if (!directory.mkdirs()) {
				ErrorsDialogUtil.getInstance().showMessage_GenericError("Cannot create results directory: " + dirName);
				return null;
			}
			return dirName;
		} catch (Exception e) {
			DialogUtil.getInstance().showMessage_ExceptionError(e);
			e.printStackTrace();
		}
		return null;
	}

}
