/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 * 
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
/*
 * generated by Xtext 2.34.0
 */
package eu.fbk.sysmlv2.tests

import com.google.inject.Inject
import org.eclipse.xtext.diagnostics.Diagnostic
import org.eclipse.xtext.testing.InjectWith
import org.eclipse.xtext.testing.extensions.InjectionExtension
import org.eclipse.xtext.testing.util.ParseHelper
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.^extension.ExtendWith
import org.eclipse.xtext.testing.validation.ValidationTestHelper
import eu.fbk.sysmlv2.sysMLv2.Model
import eu.fbk.sysmlv2.sysMLv2.SysMLv2Package
import com.google.inject.Provider
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.emf.common.util.URI
import java.nio.file.Files
import java.nio.file.Paths
import java.util.function.BiPredicate
import java.nio.file.Path
import java.nio.file.attribute.BasicFileAttributes

@ExtendWith(InjectionExtension)
@InjectWith(SysMLv2InjectorProvider)
class SysMLv2ParsingTest {
	@Inject extension ParseHelper<Model>
	@Inject extension ValidationTestHelper
	@Inject Provider<ResourceSet> resourceSetProvider

	def ResourceSet populateResourceSet() {
		val BiPredicate<Path, BasicFileAttributes> matcher = [ path, attributes |
			attributes.isRegularFile() && path.getFileName().toString().endsWith(".sysml")
		]
		resourceSetProvider.get => [ resourceSet |
			try (val pathsStream = Files.find(Paths.get("../../libs/"), Integer.MAX_VALUE, matcher))
				pathsStream.forEach[resourceSet.getResource(URI.createURI(it.toString), true)]
		]
	}

	@Test
	def void acceptEmptyModel() {
		'''package Example { }'''.parse.assertNoIssues
	}

	@Test
	def void rejectUnquotedNamespacesWithWhitespaces() {
		'''package Example System { }'''.parse.assertError(
			SysMLv2Package.Literals.PACKAGE,
			Diagnostic.SYNTAX_DIAGNOSTIC
		)
	}

	@Test
	def void testImports() {
		val first = '''
		package Vehicle {
			part def Door;
		}'''.parse
		'''
		package Car {
			private import Vehicle::Door;
			part carDoor : Door;
		}'''.parse(first.eResource.resourceSet).assertNoErrors
	}

	@Test
	def void testTransitiveImports() {
		val first = '''
		package Vehicle {
			part def Door;
		}'''.parse
		'''
		package Car {
			public import Vehicle::Door;
			part carDoor : Door;
		}'''.parse(first.eResource.resourceSet).assertNoErrors
		'''
			package Van {
				private import Car::*;
				part vanDoor : Door;
			}
		'''.parse(first.eResource.resourceSet).assertNoErrors
	}

	@Test
	def void testTransitiveImports_02() {
		val first = '''
		package Vehicle {
			part def Door;
		}'''.parse
		'''
		package Car {
			public import Vehicle::*;
			part carDoor : Door;
		}'''.parse(first.eResource.resourceSet).assertNoErrors
		'''
			package Van {
				private import Car::*;
				part vanDoor : Door;
			}
		'''.parse(first.eResource.resourceSet).assertNoErrors
	}

	@Test
	def void testTransitiveImports_03() {
		val first = '''
		package Vehicle {
			part def Door;
		}'''.parse
		'''
		package Car {
			public import Vehicle::*;
			part carDoor : Door;
			part def Trunk;
		}'''.parse(first.eResource.resourceSet).assertNoErrors
		'''
			package Van {
				public import Car::*;
			}
		'''.parse(first.eResource.resourceSet).assertNoErrors
		'''
			package Truck {
				private import Van::*;
				part truckDoor : Door;
				part truckTrunk : Trunk;
			}
		'''.parse(first.eResource.resourceSet).assertNoErrors
	}

	@Test
	def void testSpecialization() {
		'''package Vehicle {
			part def Door;
			part def TrunkDoor specializes Door;
		}'''.parse().assertNoErrors
	}

	@Test
	def void testExactMultiplicity() {
		'''package Car {
			part def Wheel;
			part wheel : Wheel[4];
		}'''.parse().assertNoErrors
	}

	@Test
	def void testRangeMultiplicity() {
		'''package Car {
			part def Door;
			part door : Door[3..5];
		}'''.parse().assertNoErrors
	}

	@Test
	def void testRedefinitionOfTypeMember() {
		'''package Car {
			part def Engine {
				part def Pipe;
				part pipe : Pipe;
			}
			part carEngine : Engine {
				part shortPipe :>> pipe;
			}
		}'''.parse().assertNoErrors
	}

	@Test
	def void testRedefinitionInSpecialization() {
		'''package Car {
			part def Engine {
				part def Pipe;
				part pipe : Pipe;
			}
			part def carEngine specializes Engine {
				part shortPipe redefines pipe;
			}
		}'''.parse().assertNoErrors
	}

	@Test
	def void testSimpleEnumDefinition() {
		'''package Test {
			enum def Points {
				enum A = 10.0;
				enum B = 9.5;
				enum C;
			}
		}'''.parse().assertNoErrors
	}

	@Test
	def void testSimpleStateMachine() {
		'''package Car {
			item def PowerOn;
			item def PowerOff;
			state EngineState {
				entry; then off;
				state off;
				transition 'off on'
				first off
				accept PowerOn then on;
				state on;
				transition shutdown
				first on accept PowerOff then off;
			}
		}'''.parse().assertNoErrors
	}

	@Test
	def void testStateMachineWithCompositeActions() {
		'''package AngleSensor {
			part def AngleSensor {
				in value;
				out sensed_value;

				state angleSensorStates {
					entry assign sensed_value := 0;
					then primary;
					state primary;

					transition 'primary-primary'
						first primary
						do action {assign sensed_value := 1;}
						then primary;
				}
			}
		}'''.parse().assertNoErrors
	}

	@Test
	def void testSSRModel() {
		'''package SSR1 {
			private import ScalarValues::Real;
			private import ScalarValues::Boolean;
		
			private import Contracts::*;
			private import LogicFunctions::*;
			private import LTLFunctions::*;
			private import ThreatsPropagation::States::*;
			private import Hierarchy::*;
		
			// Definitions
			part def System {
				in speed : Real;
				out sensed_speed: Real;
				out sensed_speed_is_present : Boolean;	
				
				part sensor1: SpeedSensor::SpeedSensor;
				part sensor2: SpeedSensor::SpeedSensor;
				
				part selector : Selector::Selector;
				
				part monitor1: MonitorPresence::MonitorPresence;
				part monitor2: MonitorPresence::MonitorPresence;
				
				bind speed = sensor1.speed;  
				bind speed = sensor2.speed;
				
				connect sensor1.sensed_speed to selector.input1;
				connect sensor2.sensed_speed to selector.input2;
				
				connect sensor1.sensed_speed_is_present to selector.input1_is_present;
				connect sensor2.sensed_speed_is_present to selector.input2_is_present;
		
				connect sensor1.sensed_speed_is_present to monitor1.input_is_present;
				connect sensor2.sensed_speed_is_present to monitor2.input_is_present;
				
				bind selector.output = sensed_speed;
				bind selector.output_is_present = sensed_speed_is_present;
				
				// Complex connections
				:>> selector.switch_current_use = monitor1.absence_alarm or monitor2.absence_alarm;
				:>> monitor1.enabled = (selector.current_use==Selector::Interval1_2::ONE);
				:>> monitor2.enabled = (selector.current_use==Selector::Interval1_2::TWO);
					
				// contract of the System component
				attribute sense : Contract 
				{
					assert constraint :>> assumption 
					{
						// assuming that:
						// - at the beginning the speed is 0
						// - the acceleration/deceleration is below a threshold
						 
						// the original one written in OCRA language
						//  speed=0 & G((next(speed) - speed)<=1 and (next(speed) - speed)>=-1)
		
						speed==0 and G({(next({speed}) - speed)<=1 and (next({speed}) - speed)>=-1 })
					}
					
					assert constraint :>> guarantee 
					{
						// we expect that:
						// - there is always a sensed speed
						// - the delta between the speed and the sensed speed is <= 4 
		
						// the original one 
						//  always ((sensed_speed - speed <= 4) and (sensed_speed - speed >= - 4) and sensed_speed_is_present)
						 
						always ({(sensed_speed - speed <= 4) and (sensed_speed - speed >= -4) and sensed_speed_is_present})
					}		
					refinedBy = (monitor1.monitor, monitor2.monitor, selector.select, selector.switch, sensor1.sense, sensor2.sense);
				}
			}	
				
			package SpeedSensor {	
				part def SpeedSensor {
					in  speed : Real;
					out sensed_speed: Real;
					out sensed_speed_is_present : Boolean;
					
					attribute sense : Contract // constraint usage
					{
						assert constraint assumption :>> assumption 
						{
							// assuming that:
							// - at the beginning the speed is 0
							// - the acceleration/deceleration is below a threshold
							 
							// the original one
							//  speed=0 & G((next(speed) - speed)<=1 and (next(speed) - speed)>=-1 )
							
							speed==0 and G({(next({speed}) - speed) <= 1 and (next({speed}) - speed)>=-1 })
						}
						
						assert constraint guarantee :>> guarantee 
						{
							// we expect that:
							// - there is always a sensed speed
							// - the delta between the speed and the sensed speed is <= 1
		
							// the original one written in OCRA language
							//  always ((sensed_speed - speed <= 1) and (sensed_speed - speed >= -1) and sensed_speed_is_present)
			   	      	      
							always({(sensed_speed - speed <= 1) and (sensed_speed - speed >= -1) and sensed_speed_is_present})  
						}		
					}
					
					state speedSensorStates
					{			
						// init transition
						entry action {assign sensed_speed := 0; assign sensed_speed_is_present := true;}  // transition effect as variables assigments
						then primary; 
						
						state primary;
							
						// nominal transition						
			            transition 'primary-primary'
				            first primary
				            if true // guard
				            do action {assign sensed_speed_is_present := true; assign sensed_speed := speed;} // effect
				            then primary;
					}
		
					state faultSpeedSensorStates
					{			
						entry action {}
						then nominal;
					
						state nominal;
						state error : Inverted	// flipped output (for Boolean signals)
						{
							:>> probability = 0.05;
							:>> property = sensed_speed_is_present;
						}
											
						transition 'nominal-error'
							first nominal
							if true // guard
							do action {} //effect
							then error;	       
					}
				}			
			}
			
			package MonitorPresence {
				part def MonitorPresence {
					in input_is_present: Boolean;
					in enabled: Boolean;
					out absence_alarm: Boolean;
					
					attribute monitor : Contract
					{
						assert constraint assumption :>> assumption 
						{
							// assuming that: at the beginning the sensor associated to this monitor is working
							
							// the original one written in OCRA language
							//  input_is_present=true
							
							(input_is_present==true)
						}
						
						assert constraint guarantee :>> guarantee 
						{
							// we expect that: an alarm is triggered whenever the monitor is enabled and the input is not present (is absent)
							
							// the original one written in OCRA language
							// always((absence_alarm) iff (enabled and not(input_is_present)))
							 
							// always({iff({next({absence_alarm})}, {enabled and not(input_is_present)})})	// correct
							always({iff({absence_alarm}, {enabled and not(input_is_present)})}) 			// wrong
						}		
					}
													
					state monitorPresenceStates 	
					{			
						// init transition				
						entry action {assign absence_alarm := false;} // transition effect as variables assigments
						then primary; 
						
						state primary;
						
						// nominal transition						
			            transition 'primary-primary'
				            first primary
				            if true // guard
				            do action {assign absence_alarm := (not(input_is_present) and (enabled));} // effect
				            then primary;
					}
					
					state faultMonitorStates
					{			
						entry action {}
						then nominal;
					
						state nominal;
						state error : StuckAtFixed	// stuck at fixed random value
						{
							:>> probability = 0.05;
							:>> property = absence_alarm;
						}
											
						transition 'nominal-error'
							first nominal
							if true // guard
							do action {} //effect
							then error;	       
					}
				}
			}
			
			package Selector {
				part def Selector {
					in input1: Real;
					in input1_is_present: Boolean;
					in input2: Real;
					in input2_is_present: Boolean;
					
					out output: Real;
					out output_is_present: Boolean;
					
					in switch_current_use: Boolean;
					out current_use: Interval1_2;
					
					attribute select : Contract
					{
						assert constraint assumption :>> assumption 
						{
							true
						}
						
						assert constraint guarantee :>> guarantee 
						{
							// we expect that:
							// - at the beginning the sensed speed is 0
							// - the sensed speed will be measured by the current sensor
							 	
							// the original one written in OCRA language
							//  (output=0 and output_is_present = TRUE) and always ((next(current_use=1) implies 
			    	 	    //  (next(output)=input1 and next(output_is_present)=input1_is_present)) and 
			   	    		//  (next(current_use=2) implies 
				     		//  (next(output)=input2 and next(output_is_present)=input2_is_present)))
				     			 
				     		(output==0 and output_is_present == true) and always ({(next({current_use==Interval1_2::ONE}) implies 
			    	 	      (next({output})==input1 and next({output_is_present})==input1_is_present)) and 
			   	    		  (next({current_use==Interval1_2::TWO}) implies 
				     		  (next({output})==input2 and next({output_is_present})==input2_is_present))})	     		  
						}		
					}
					 
					attribute switch : Contract
					{
						assert constraint assumption :>> assumption 
						{
							true
						}
						
						assert constraint guarantee :>> guarantee 
						{
							// we expect that: the switch of the sensor depends only on the input boolean port 'switch_current_use'
							
							// the original one written in OCRA language
							//  always (
			    			//  ((current_use=1 and switch_current_use) implies next(current_use)=2) and
						    //  ((current_use=2 and switch_current_use) implies next(current_use)=1) and
						    //  ((not switch_current_use) implies not change(current_use)))
						     				     
						    always ({
						    ((current_use==Interval1_2::ONE and switch_current_use) implies next({current_use})==Interval1_2::TWO) and
						    ((current_use==Interval1_2::TWO and switch_current_use) implies next({current_use})==Interval1_2::ONE) and
						    ((not switch_current_use) implies not change({current_use}))})			    
						}		
					}
					
					state selectorStates 	
					{			
						// init transition				
						entry action {assign current_use := Interval1_2::ONE; assign output := 0; assign output_is_present := true;} // transition effect as variables assigments
						then input_1; 
						
						state input_1;
						state input_2;
							
						// nominal transition						
			            transition 'input_1-input_1'
				            first input_1
				            if not switch_current_use // guard
				            do action {assign output := input1 ; assign output_is_present := input1_is_present;} // effect
				            then input_1;
				            
				        // nominal transition    
				        transition 'input_1-input_2'
				            first input_1
				            if switch_current_use // guard
				            do action {assign current_use := Interval1_2::TWO; assign output := input2; assign output_is_present := input2_is_present;} // effect
				            then input_2; 
				         
				        // nominal transition    
				        transition 'input_2-input_2'
				            first input_2
				            if not switch_current_use // guard
				            do action {assign output := input2; assign output_is_present := input2_is_present;} //effect
				            then input_2;
				          
				        // nominal transition    
				        transition 'input_2-input_1'
				            first input_2
				            if switch_current_use // guard
				            do action {assign current_use := Interval1_2::ONE; assign output := input1 ; assign output_is_present := input1_is_present;} // effect
				            then input_1;          
					}
					
					state faultSelectorStates
					{			
						entry action {}
						then nominal;
					
						state nominal;
						state error : StuckAtByReference_D	// stuck at given term
						{
							:>> probability = 0.05;
							:>> property = output_is_present;
							:>> term = "FALSE";
						}
											
						transition 'nominal-error'
							first nominal
							if true // guard
							do action {} //effect
							then error;	       
					}
				}
				
				enum def Interval1_2 {
					enum ONE;
					enum TWO;
				}
			}
				
			// Usage level: system decomposition and parts are interconnected
			part ssr : System, SystemComponent {			
				
						 		
			}
		}
		
		// For model checking, use: 
		//   Selector component + ltlspec + ic3 F(current_use=TWO)			// False, current_use can be never TWO
		//   Selector component + ltlspec + ic3 G(current_use=TWO)			// perch� vedo la traccia? Non � sempre valida?
		//   System component + ltlspec + ic3 F(selector.current_use=TWO)	// False, current_use can be never TWO
		//   Selector component + ltlspec + ic3 G(State!=input_2)			// False, input_2 can be reached, no contraints on inputs
		//   System component + ltlspec + ic3 G(selector.State!=input_2)	// True, input_2 cannot be reached because there no faults
		// 
		// For FTA and FMEA, use msat and the following TLE/properties:
		//   sensed_speed_is_present=FALSE
		//   sensor1.sensed_speed_is_present=FALSE
		// 
		// For FDI, use msat_bmc (bmc for optimization):
		//   CONDITION: System.sensor1.faultSpeedSensorStates.mode != NOMINAL or = error#FAULT
		//   TYPE: finite'''.parse(populateResourceSet).assertNoErrors
	}

	@Test
	def void testDocumentation() {
		'''package 'Documentation Example' {
			doc /* This is documentation of the owning
				 * package.
				 */
			part def Automobile {
				doc Document1
					/* This is documentation of Automobile. */
				}
		}'''.parse.assertNoErrors
	}

	@Test
	def void acceptPortTypings() {
		'''package Example {
			port def MyPort;
			port input : MyPort;
			port output : ~MyPort;
		}'''.parse.assertNoIssues
	}

	@Test
	def void acceptSimpleAction() {
		'''package Camera {
			attribute def Scene;
			attribute def Image;
			action def Focus { in scene : Scene; out image: Image; }
			attribute def Picture;
			action def Shoot { in image : Image; out picture : Picture; }
			action def TakePicture{in scene : Scene; out picture : Picture;}
			action takePicture : TakePicture {
				in scene;
				action focus : Focus {
					in scene = takePicture.scene;
					out image;
				}
				then action shoot : Shoot {
					in image;
					out picture;
				}
				out picture = shoot.picture;
			}
		}'''.parse.assertNoIssues
	}

	@Test
	def void complexScopingTest() {
		val first = '''package Example {
			public import ScalarValues::*;
			part def MyPart {
				port def MyPort {
					in attribute attr : Integer := 10;
				}
				dummy;
			}
		}'''.parse(populateResourceSet)
		'''
		package AnotherExample {
			private import Example::MyPart;
			part component : MyPart {
				port def SpecialPort specializes ~MyPort;
				port myPort : SpecialPort;
				attribute nine : Real;
			}
			bind component.myPort.attr = MyPart::dummy;
			:>> component.nine : Integer = 9;
		}
		'''.parse(first.eResource.resourceSet)
	}
}
