/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.sysmlv2;

import java.io.File;
import java.net.URL;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.osgi.service.datalocation.Location;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;

import com.google.common.collect.ImmutableList;

public class SysMLv2Activator implements BundleActivator {

	private static final Set<ResourceSet> POPULATED_RESOURCE_SETS = new HashSet<>();

	private static final String SYSML_EXTENSION = ".sysml";

	private static IWorkspaceRoot WORKSPACE_ROOT;

	private static ImmutableList<URI> FILES;

	/** The default workspace to use. */
	private static final String WORKSPACE_PATH = "./model";

	@Override
	public void start(BundleContext context) throws Exception {
		setWorkspaceIfUnset();
		WORKSPACE_ROOT = ResourcesPlugin.getWorkspace().getRoot();
		FILES = collectFilesWithExtension(WORKSPACE_ROOT, SYSML_EXTENSION, ImmutableList.builder());
		IResourceChangeListener rcl = new ProjectListener();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(rcl, IResourceChangeEvent.POST_CHANGE);
		WORKSPACE_ROOT.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
	}

	@Override
	public void stop(BundleContext context) throws Exception {
	}

	private static void setWorkspaceIfUnset() {
		// Check if the workspace is already set
		Location instanceLocation = Platform.getInstanceLocation();
		if (!instanceLocation.isSet()) {
			// Attempt to set the workspace location
			File workspaceDirectory = new File(WORKSPACE_PATH);
			if (!workspaceDirectory.exists()) {
				workspaceDirectory.mkdirs();
			}
			try {
				URL workspaceURL = workspaceDirectory.toURI().toURL();
				if (!instanceLocation.set(workspaceURL, false)) {
					System.err.println("Failed to set workspace location: " + workspaceURL);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
			System.out.println("Setting workspace to " + instanceLocation.getURL());
		} else {
			System.out.println("Workspace already set to " + instanceLocation.getURL());
		}
	}

	private static ImmutableList<URI> collectFilesWithExtension(IResource resource, String fileExtension, ImmutableList.Builder<URI> builder) {
		if (resource instanceof final IContainer container) {
			if (container instanceof IProject project && !project.isOpen()) return null;
			try {
				IResource[] members = container.members();
				for (IResource member : members) {
					if (member instanceof final IFolder folder && !folder.getName().equals("bin")) {
						collectFilesWithExtension(folder, fileExtension, builder);
					} else if (!(member instanceof IFolder)) {
						collectFilesWithExtension(member, fileExtension, builder);
					}
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
		} else if (resource instanceof final IFile file && file.getName().endsWith(fileExtension)) {
			builder.add(URI.createFileURI(file.getRawLocation().makeAbsolute().toOSString()));
		}

		if (resource.equals(WORKSPACE_ROOT)) {
			return builder.build();
		} else {
			return null;
		}
	}

	public static void populateResourceSet(final ResourceSet resourceSet) {
		if (POPULATED_RESOURCE_SETS.add(resourceSet)) {
			FILES.forEach(file -> resourceSet.getResource(file, true));
		}
	}

	private final class ProjectListener implements IResourceChangeListener {
		@Override
		public void resourceChanged(IResourceChangeEvent event) {
			if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
				FILES = collectFilesWithExtension(WORKSPACE_ROOT, SYSML_EXTENSION, ImmutableList.builder());
				POPULATED_RESOURCE_SETS.clear();
			}
		}
	}

}
