/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.sysmlv2.scoping;

import static java.util.Collections.singletonList;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescriptionsProvider;
import org.eclipse.xtext.resource.ISelectable;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.impl.ImportNormalizer;
import org.eclipse.xtext.scoping.impl.ImportedNamespaceAwareLocalScopeProvider;

import com.google.inject.Inject;

import eu.fbk.sysmlv2.naming.SysMLv2QualifiedNameConverter;
import eu.fbk.sysmlv2.services.SysMLv2ValueConverter;
import eu.fbk.sysmlv2.sysMLv2.Import;
import eu.fbk.sysmlv2.sysMLv2.ImportType;
import eu.fbk.sysmlv2.sysMLv2.PortDefinition;
import eu.fbk.sysmlv2.sysMLv2.SysMLv2Package;

public class SysMLv2ImportedNamespaceAwareLocalScopeProvider extends ImportedNamespaceAwareLocalScopeProvider {

	@Inject
	private IResourceDescriptionsProvider resourceDescriptionsProvider;

	@Inject
	private SysMLv2QualifiedNameConverter qualifiedNameConverter;

	@Inject
	private SysMLv2ValueConverter valueConverter;

	private boolean isRootCall = true;

	@Override
	protected List<ImportNormalizer> getImportedNamespaceResolvers(final EObject context, boolean ignoreCase) {
		return EcoreUtil2.typeSelect(context.eContents(), Import.class).stream()
				.filter(importStatement -> isRootCall || importStatement.getType().equals(ImportType.PUBLIC))
				.map(importStatement ->
						createImportedNamespaceResolver(getImportedNamespace(importStatement), ignoreCase))
				.filter(Objects::nonNull)
				.toList();
	}

	private void recursivelyComputeTransitiveImports(EObject context, boolean ignoreCase, Set<ImportNormalizer> intermediate) {
		final List<ImportNormalizer> importNormalizers = getImportedNamespaceResolvers(context, ignoreCase);
		intermediate.addAll(importNormalizers);
		final List<EObject> importedEObjects = new ArrayList<>();
		final ResourceSet resourceSet = EcoreUtil2.getResourceSet(context);
		importNormalizers.forEach(normalizer -> {
			final QualifiedName importedName = normalizer.getImportedNamespacePrefix();
			final Iterator<IEObjectDescription> descriptions = resourceDescriptionsProvider.getResourceDescriptions(resourceSet)
				.getExportedObjects(SysMLv2Package.Literals.PACKAGE_MEMBER, importedName, ignoreCase)
				.iterator();
			if (!descriptions.hasNext()) {
				return;
			}
			final IEObjectDescription description = descriptions.next();
			EObject eObject = description.getEObjectOrProxy();
			if (eObject.eIsProxy()) {
				eObject = EcoreUtil.resolve(eObject, resourceSet);
			}
			if (eObject instanceof final PortDefinition portDefinition) {
				final String conjugatedName = "~" + portDefinition.getName();
				final QualifiedName conjugateQualifiedName = importedName.skipLast(1).append(conjugatedName);
				final ImportNormalizer conjugateNormalizer = new SysMLv2ImportNormalizer(conjugateQualifiedName, false, ignoreCase);
				intermediate.add(conjugateNormalizer);
			}
			importedEObjects.add(eObject);
		});
		isRootCall = false;
		importedEObjects.forEach(e -> recursivelyComputeTransitiveImports(e, ignoreCase, intermediate));
	}

	@Override
	protected String getImportedNamespace(EObject context) {
		if (context instanceof final Import importStatement) {
			QualifiedName qualifiedName = null;
			final var nodes = NodeModelUtils.findNodesForFeature(importStatement, SysMLv2Package.Literals.IMPORT__IMPORTED_NAMESPACE);
			if (!nodes.isEmpty()) {
				String importedNamespaceString = NodeModelUtils.getTokenText(nodes.get(0));
				importedNamespaceString = valueConverter.QualifiedName().toValue(importedNamespaceString, null);
				if (!importedNamespaceString.isEmpty()) {
					qualifiedName = qualifiedNameConverter.toQualifiedName(importedNamespaceString);
				}
			}
			if (qualifiedName != null) {
				final String wildcardSegment = qualifiedNameConverter.getDelimiter() + getWildCard();
				return importStatement.isWildcard() ? qualifiedNameConverter.toString(qualifiedName) + wildcardSegment : qualifiedNameConverter.toString(qualifiedName);
			}
		}
		return null;
	}

	@Override
	protected IScope getLocalElementsScope(IScope parent, final EObject context,
			final EReference reference) {
		IScope result = parent;
		ISelectable allDescriptions = getAllDescriptions(context.eResource());
		QualifiedName name = getQualifiedNameOfLocalElement(context);
		boolean ignoreCase = isIgnoreCase(reference);
		isRootCall = true;

		final Set<ImportNormalizer> intermediate = new HashSet<>();
		recursivelyComputeTransitiveImports(context, ignoreCase, intermediate);
		final List<ImportNormalizer> namespaceResolvers = new ArrayList<>(intermediate);

		if (!namespaceResolvers.isEmpty()) {
			if (isRelativeImport() && name != null && !name.isEmpty()) {
				ImportNormalizer localNormalizer = doCreateImportNormalizer(name, true, ignoreCase);
				result = createImportScope(result, singletonList(localNormalizer), allDescriptions, reference.getEReferenceType(), ignoreCase);
			}
			result = createImportScope(result, namespaceResolvers, null, reference.getEReferenceType(), ignoreCase);
		}
		if (name != null) {
			ImportNormalizer localNormalizer = doCreateImportNormalizer(name, true, ignoreCase);
			result = createImportScope(result, singletonList(localNormalizer), allDescriptions, reference.getEReferenceType(), ignoreCase);
		}
		return result;
	}

	@Override
	protected ImportNormalizer doCreateImportNormalizer(QualifiedName importedNamespace, boolean wildcard, boolean ignoreCase) {
		return new SysMLv2ImportNormalizer(importedNamespace, wildcard, ignoreCase);
	}

}
