/*
 * SPDX-FileCopyrightText: 2024 Fondazione Bruno Kessler
 *
 * SPDX-FileContributor: Tommaso Fonda - initial API and implementation
 */
package eu.fbk.sysmlv2.services;

import static java.util.function.Predicate.not;
import static org.eclipse.xtext.EcoreUtil2.eAllContentsAsList;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.stream.Stream;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.resource.DerivedStateAwareResource;
import org.eclipse.xtext.resource.IDerivedStateComputer;

import eu.fbk.sysmlv2.sysMLv2.Container;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.Direction;
import eu.fbk.sysmlv2.sysMLv2.EntryTransition;
import eu.fbk.sysmlv2.sysMLv2.PortDefinition;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.SysMLv2Package;
import eu.fbk.sysmlv2.util.SysMLv2Util;

public class SysMLv2DerivedStateComputer implements IDerivedStateComputer {

	@Override
	public void installDerivedState(DerivedStateAwareResource resource, boolean preLinkingPhase) {
		installConjugatedPortDefinitions(resource);
		installEntryTransitions(resource);
	}

	@Override
	public void discardDerivedState(DerivedStateAwareResource resource) {
		discardConjugatedPortDefinitions(resource);
		discardEntryTransitions(resource);
	}

	private void installConjugatedPortDefinitions(DerivedStateAwareResource resource) {
		getExplicitPortDefinitions(resource).forEach(portDefinition -> {
			final PortDefinition conjugated = EcoreUtil.copy(portDefinition);
			conjugated.setName("~" + portDefinition.getName());
			typeSelect(conjugated.getMembers(), DirectedElement.class).forEach(member -> {
				if (Direction.IN.equals(member.getDirection())) {
					member.setDirection(Direction.OUT);
				} else if (Direction.OUT.equals(member.getDirection())) {
					member.setDirection(Direction.IN);
				}
			});
			portDefinition.setConjugatedPortDefinition(conjugated);
			conjugated.setConjugatedPortDefinition(portDefinition);
			final Container container = (Container) portDefinition.eContainer();
			container.getMembers().add(conjugated);
		});
	}

	private void installEntryTransitions(DerivedStateAwareResource resource) {
		typeSelect(eAllContentsAsList(resource), StateUsage.class)
				.forEach(stateUsage -> EcoreUtil2.getAllContentsOfType(stateUsage, EntryTransition.class)
						.forEach(transition -> transition.setSource(stateUsage)));
	}

	private void discardConjugatedPortDefinitions(DerivedStateAwareResource resource) {
		getExplicitPortDefinitions(resource).forEach(portDefinition -> {
			final Container container = (Container) portDefinition.eContainer();
			container.getMembers().removeIf(eObject -> eObject instanceof final PortDefinition portDef
					&& SysMLv2Util.isConjugatedPort(portDef));
			portDefinition.eUnset(SysMLv2Package.Literals.PORT_DEFINITION__CONJUGATED_PORT_DEFINITION);
		});
	}

	private void discardEntryTransitions(DerivedStateAwareResource resource) {
		typeSelect(eAllContentsAsList(resource), EntryTransition.class).stream()
			.forEach(transition -> transition.setSource(null));
	}

	private static Stream<PortDefinition> getExplicitPortDefinitions(DerivedStateAwareResource resource) {
		return typeSelect(eAllContentsAsList(resource), PortDefinition.class).stream()
				.filter(not(SysMLv2Util::isConjugatedPort));
	}


}
