package eu.fbk.sysmlv2.services;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.eclipse.xtext.common.services.DefaultTerminalConverters;
import org.eclipse.xtext.conversion.IValueConverter;
import org.eclipse.xtext.conversion.ValueConverter;
import org.eclipse.xtext.conversion.ValueConverterException;
import org.eclipse.xtext.nodemodel.INode;

public class SysMLv2ValueConverter extends DefaultTerminalConverters {

	private final IValueConverter<String> nameValueConverter = new IValueConverter<>() {
		@Override
		public String toValue(String string, INode node) {
			if (string != null && string.startsWith("'") && !string.contains(" ")) {
				return string.substring(1, string.length() - 1);
			} else {
				return string;
			}
		}

		@Override
		public String toString(String value) {
			if (value != null && !value.startsWith("'") && value.contains(" ")) {
				return "'" + value + "'";
			} else {
				return value;
			}
		}
	};

	@ValueConverter(rule = "Name")
	public IValueConverter<String> Name() {
		return nameValueConverter;
	}

	@ValueConverter(rule = "QualifiedName")
	public IValueConverter<String> QualifiedName() {
		return new IValueConverter<>() {
			@Override
			public String toValue(String string, INode node) {
				return Arrays.stream(string.split("::")).map(name -> nameValueConverter.toValue(name, node)).collect(Collectors.joining("::"));
			}

			@Override
			public String toString(String value) {
				return Arrays.stream(value.split("::")).map(nameValueConverter::toString).collect(Collectors.joining("::"));
			}
		};
	}

	@ValueConverter(rule = "MultiplicityBound")
	public IValueConverter<Integer> MultiplicityBound() {
		return new IValueConverter<>() {
			private static final String WILDCARD = "*";

			@Override
			public Integer toValue(String string, INode node) throws ValueConverterException {
				if (string == null) {
					return null;
				} else if (WILDCARD.equals(string.trim())) {
					return -1;
				} else {
					try {
						return Integer.parseInt(string);
					} catch (NumberFormatException e) {
						throw new ValueConverterException("Couldn't convert \"" + string + "\" to int", node, e);
					}
				}
			}

			@Override
			public String toString(Integer value) {
				if (value == -1) {
					return WILDCARD;
				} else {
					return Integer.toString(value);
				}
			}
		};
	}

	@ValueConverter(rule = "STRING_VALUE")
	public IValueConverter<String> STRING_VALUE() {
		return new IValueConverter<>() {
			@Override
			public String toValue(String string, INode node) {
				if (string == null) {
					return null;
				} else {
					return string.substring(1, string.length() - 1);
				}
			}

			@Override
			public String toString(String value) {
				if (value == null) {
					return null;
				} else {
					return "\"" + value + "\"";
				}
			}
		};
	}

}
