package eu.fbk.sysmlv2.util;

import static java.util.function.Predicate.not;
import static org.eclipse.xtext.EcoreUtil2.typeSelect;

import java.util.List;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtext.EcoreUtil2;

import eu.fbk.sysmlv2.sysMLv2.ActionUsage;
import eu.fbk.sysmlv2.sysMLv2.AttributeUsage;
import eu.fbk.sysmlv2.sysMLv2.Container;
import eu.fbk.sysmlv2.sysMLv2.Definition;
import eu.fbk.sysmlv2.sysMLv2.DirectedElement;
import eu.fbk.sysmlv2.sysMLv2.ItemDefinition;
import eu.fbk.sysmlv2.sysMLv2.ItemUsage;
import eu.fbk.sysmlv2.sysMLv2.MetadataUsage;
import eu.fbk.sysmlv2.sysMLv2.Model;
import eu.fbk.sysmlv2.sysMLv2.NamedElement;
import eu.fbk.sysmlv2.sysMLv2.Package;
import eu.fbk.sysmlv2.sysMLv2.PartDefinition;
import eu.fbk.sysmlv2.sysMLv2.PartUsage;
import eu.fbk.sysmlv2.sysMLv2.PortDefinition;
import eu.fbk.sysmlv2.sysMLv2.PortUsage;
import eu.fbk.sysmlv2.sysMLv2.ReferenceExpression;
import eu.fbk.sysmlv2.sysMLv2.StateUsage;
import eu.fbk.sysmlv2.sysMLv2.Transition;
import eu.fbk.sysmlv2.sysMLv2.TypedElement;
import eu.fbk.sysmlv2.sysMLv2.Usage;

public class SysMLv2Util {

	private static final String SYSTEM_COMPONENT = "SystemComponent";
	private static final String DURATIVE_TRANSITION = "DurativeTransition";
	private static final String ACTIVITY = "Activity";
	private static final String SUBSYSTEM = "Subsystem";
	private static final String EVENT = "Event";
	private static final String IS_OBSERVABLE = "IsObservable";

	// Suppress default constructor for noninstantiability
	private SysMLv2Util() {
		throw new AssertionError();
	}

	public static boolean isDurative(Transition transition) {
		final Definition type = getType(transition);
		return type != null && DURATIVE_TRANSITION.equals(type.getName());
	}

	public static boolean isConjugatedPort(final PortDefinition portDefinition) {
		return portDefinition.getName() != null && portDefinition.getName().startsWith("~");
	}

	public static <T extends NamedElement> T getMemberByName(Container container, Class<T> type, String name) {
		return EcoreUtil2.typeSelect(container.getMembers(), type).stream()
			.filter(element -> name.equals(element.getName()))
			.findAny().orElse(null);
	}

	public static String getSpecializedDefinitionName(Definition definition) {
		if (definition.getSpecialization() != null &&
				!definition.getSpecialization().getReferences().isEmpty() &&
				definition.getSpecialization().getReferences().get(0).getReferencedElement() != null) {
			return definition.getSpecialization().getReferences().get(0).getReferencedElement().getName();
		} else {
			return null;
		}
	}

	public static boolean partDefinitionIsActivity(PartDefinition partDefinition) {
		return ACTIVITY.equals(getSpecializedDefinitionName(partDefinition));
	}

	public static boolean partDefinitionIsSubsystem(PartDefinition partDefinition) {
		return SUBSYSTEM.equals(getSpecializedDefinitionName(partDefinition));
	}

	public static boolean itemDefinitionIsEvent(ItemDefinition itemDefinition) {
		return EVENT.equals(getSpecializedDefinitionName(itemDefinition));
	}

	public static boolean itemUsageIsEvent(ItemUsage itemUsage) {
		final ItemDefinition itemDefinition = (ItemDefinition) getType(itemUsage);
		return EVENT.equals(itemDefinition.getName()) || itemDefinitionIsEvent(itemDefinition);
	}

	public static List<Definition> getAllTypes(TypedElement element) {
		if (element.getTyping() != null) {
			return typeSelect(element.getTyping().getReferences().stream()
					.map(ReferenceExpression::getReferencedElement).toList(), Definition.class);
		} else {
			return List.of();
		}
	}

	public static Definition getType(TypedElement element) {
		final List<Definition> allTypes = getAllTypes(element);
		return !allTypes.isEmpty() ? allTypes.get(0) : null;
	}

	public static boolean isStateMachine(StateUsage stateUsage) {
		return stateUsage.getMembers().stream().anyMatch(StateUsage.class::isInstance);
	}

	public static boolean isEvent(Usage usage) {
		return isEventDefinition(getType(usage));
	}

	public static boolean isEventDefinition(Definition definition) {
		if (definition instanceof final ItemDefinition itemDefinition) {
			if (itemDefinition.getName().equals(EVENT)) {
				return true;
			}
			return EVENT.equals(getSpecializedDefinitionName(itemDefinition));
		}
		return false;
	}

	public static DirectedElement getDataMemberFromPort(PortUsage portUsage) {
		final PortDefinition portDefinition = (PortDefinition) getType(portUsage);
		return portDefinition.getMembers().stream()
			.filter(DirectedElement.class::isInstance).map(DirectedElement.class::cast)
			.filter(not(SysMLv2Util::isEvent))
			.findAny().orElseThrow(() -> new RuntimeException("No data type member in port definition " + portDefinition.getName()));
	}

	public static DirectedElement getEventMemberFromPort(PortUsage portUsage) {
		final PortDefinition portDefinition = (PortDefinition) getType(portUsage);
		return portDefinition.getMembers().stream()
			.filter(DirectedElement.class::isInstance).map(DirectedElement.class::cast)
			.filter(SysMLv2Util::isEvent)
			.findAny().orElseThrow(() -> new RuntimeException("No event type member in port definition " + portDefinition.getName()));
	}

	public static boolean resourceBelongsToLibrary(Resource resource) {
		return resource.getContents().get(0) instanceof final Model model
				&& model.getMembers().get(0) instanceof final Package pkg && pkg.isLibrary();
	}

	public static boolean containsMetadataUsagesOfType(Container container, String typeName) {
		return typeSelect(container.getMembers(), MetadataUsage.class).stream()
				.anyMatch(metadata -> typeName.equals(getType(metadata).getName()));
	}

	public static boolean isObservableVariable(AttributeUsage variable) {
		return containsMetadataUsagesOfType(variable, IS_OBSERVABLE);
	}

	private static boolean isTypeSystemComponent(Definition definition) {
		return SYSTEM_COMPONENT.equals(definition.getName())
				|| SYSTEM_COMPONENT.equals(getSpecializedDefinitionName(definition));
	}

	public static boolean isSystemPartUsage(PartUsage partUsage) {
		return getAllTypes(partUsage).stream().anyMatch(SysMLv2Util::isTypeSystemComponent);
	}

	public static boolean isSetterAction(ActionUsage actionUsage) {
		final var type = SysMLv2Util.getType(actionUsage);
		return type != null && "Setter".equals(type.getName());
	}

}
